package Truss is
   ------------------------------------------------------------
   -- This package provides procedures to solve simple trusses
   -- The trusses must be statically determinate, consisting of a
   -- pin joint and a roller joint.  The solver will habdle a maximum of
   -- ten joints, 20 bars, and 10 loads.  The loads must be suspended from 
   -- a joint, but can be in any direction.
   -- Author: Howard Kleinwaks, for use in Unified Engineering at MIT, 2003
   -- Last Modified: October 12, 2003
   --------------------------------------------------------------


   --declare variables to be used in package

   type Point is 
      record 
         X : Float;  
         Y : Float;  
      end record; 
   Max_Joints      : constant Integer := 10; --maximum number of joints in truss   
   Max_Connections : constant Integer := 10; --maximum number of connections to any one joint   
   Max_Loads       : constant Integer := 10; --maximum number of loads on the truss   
   Max_Bars        : constant Integer := 20; -- maximum number of bars in the truss   

   type Connectarray is array (1..Max_Connections) of Integer; -- array of bars connected to any one joint

   type Joint is --record for storing joints
      record 
         Location         : Point; --x and y location of the joint  
         Num_Connect_Bars : Integer;  -- number of bars connecting to the joint
         Bar_Array        : Connectarray;  --array of connecting bars
      end record; 
   type Jointarray is array (1..Max_Joints) of Joint; --array of joints

   type Force is --record to hold bar forces and reaction forces
      record 
         Force      : Float;  --the bar or reaction force
         Joint1     : Integer;  --the first joint that the bar is attached to
         Joint2     : Integer;  --the second joint that the bar is attached to
         Angle_1_2  : Float;  --angle the bar makes with the horizontal when going from joint 1 to joint 2
         Angle_2_1  : Float;  --angle bar makes with horizontal when going from joint 2 to joint 1
      end record; 
   type Forcearray is array (1..Max_Bars) of Force; --array of forces 
      
   type Direction is --enumerated type to allow user to specify direction of roller reaction force 
         (Horiz, 
          Vert); 
   type Load is --record to hold the load 
      record 
         Joint : Integer;  --joint where load is applied
         Mag   : Float;  --magnitude of the load
         Angle : Float;  --angle that the load makes with the horizontal
      end record; 
   type Loadarray is array (1..Max_Loads) of Load; 

   procedure Truss_Input (
         Joints                 :    out Jointarray; 
         Loads                  :    out Loadarray;  
         Num_Loads              :    out Integer;    
         Num_Bars               :    out Integer;    
         Num_Joints             :    out Integer;    
         Pin_Joint_Number       :    out Integer;    
         Roller_Joint_Number    :    out Integer;    
         Roller_Joint_Direction :    out Direction   ); 
   --procedure to input the parameters of the truss from a file, called "Truss_Input.txt"
   --all the inputs are assigned values, and the array of joints is set up
   
   procedure Force_Set_Up (
         Num_Joints             : in     Integer;    
         Num_Bars               : in     Integer;    
         Joints                 : in     Jointarray; 
         Forces                 :    out Forcearray; 
         Pin_Joint_Number       : in     Integer;    
         Roller_Joint_Number    : in     Integer;    
         Roller_Joint_Direction : in     Direction   ); 
   -- Procedure to set up the force array to be ready to solve for the forces.
   -- all the end joints and the angles are set
   procedure Solve_Forces (
         Forces     : in out Forcearray; 
         Num_Bars   : in     Integer;    
         Num_Joints : in     Integer;    
         Loads      : in     Loadarray;  
      Num_Loads  : in     Integer     );
      --procedure to solve for the forces in each bar, setting the value in the force array
   procedure Truss_Output (
         Forces   : Forcearray; 
         Num_Bars : Integer     ); 
   -- procedure to output the results.

end Truss;