%references:
%SMAD
% Holman, J.P. Heat Transfer.  New York:  McGraw-Hill, Inc. 1997.

function feasible = thermal(alpha, epsilon, T_user_max, T_user_min, interior_vol, thickness, altitude, Q_w)

% thermal - checks to see if spacecraft design is feasible from a thermal
% standpoint for a first-order approximation

% Assumptions:
% - isothermal sphere
% - uniform electrical generation on sphere
% - material may be two one, two, or more layers thick, but usually
% still use epsilon to distinguish multiple layer insulation since radiation is extremely predominant 
% over conduction in the space environment (SMAD pg451)
% - takes into account one half always in sun and view factor from Earth
% - no attempt to mainain spacecraft battery limits, possibly separate
% control system with thermostats

%Inputs:

% alpha - solar absorptivity of the sphere
% epsilon - IR emissivity of the sphere
% T_user_max [K] - worst-case hot temperature for equipment
% T_user_min [K] - worst-case cold temparture for equipment
% interior_vol [m^3] - interior volume of satellite
% thickness [m] - thickness of structural and thermal material
% altitude [m] - height of spacecraft from earth's surface
% Q_w [W] - electrical power disappation (if not known use orbital average
% power

%Outputs
% feasible - 0 if not feasible, 1 if feasible

%Constants:
G_s = 1367; %[W.m^-2] solar constant
a = 0.3
; %albedo 30% of direct solar
q_i = 237; %[W.m^-2] earth IR emission
sigma = 5.67051e-8; %[W.m^-2.K-4] Stefan-Boltzmann's constant
R_e = 6378e3; %[m] - radius of earth
rho = asin(R_e/(altitude+R_e));
K_a = 0.664 + 0.521*rho - 0.203*rho^2; % a factor which accounts for the reflection of collimated incoming solar energy off a spherical Earth
D = (6*interior_vol/pi)^(1/3) + 2*thickness; %[m] calculation of outer diameter of satellite
A = pi*D^2; %[m^2] surface area of satellite, m^2
A_c = A/4; %[m^2] cross section area of the spherical satellite.
F = (1 - cos(rho))/2;

% Calculations for temperatures from SMAD pg447

T_max_s = ((A_c*G_s*alpha + A*F*q_i*epsilon + A*F*G_s*a*alpha*K_a + Q_w)/(A*sigma*epsilon))^(0.25)
T_min_s = ((A*F*q_i*epsilon + Q_w)/(A*sigma*epsilon))^(0.25)

% Calculations for space radiator
% Assume theta = 90 degrees since we will place space radiator not facing
% sun
% Assume radiator does not absorb IR emission from earth or albedo from
% earth.

T_ave = (T_user_max + T_user_min)/2;  % use average case temperature
A_r = Q_w/(sigma*epsilon*(T_ave)^4)

% Check for feasiblity
if T_user_max > T_max_s & T_user_min < T_min_s & A_r < A
    feasible = 1;
else
    feasible = 0;
end


