% William Nadir
% 16.851 Satellite Engineering
% Problem Set 5
%
% Space Hotel Structural Design Software Module
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% INPUTS
%
% N_guests  = Number of people staying on board the space hotel at any time
% duration  = Duration of stay for guests of the space hotel (days)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% OUTPUTS
%
% str_mass  = Mass estimate of structure of space hotel (kg)
% spin_rate = Spin rate of hotel to produce artificial gravity in
%             habitation modules (m/s)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [str_mass,spin_rate] = structure(N_guests, duration)

% Here the volume required per passenger onboard the hotel is determined
time        = [2 30 60]; % duration of stay (days)
req_volume  = [150 780 1800]; % required "free volume" per passenger (ft^3)

if duration <= time(1)
    vol = req_volume(1);
elseif duration > time(1) && duration < time(2)
    vol = interp1(time,req_volume,duration,'linear');
elseif duration == time(2)
    vol = req_volume(2);
elseif duration > time(2) && duration < time(3)
    vol = interp1(time,req_volume,duration,'linear');
else
    vol = req_volume(3);
end

% Input habitation module volume here
hab_free_volume = 4500; % (ft^3)

% Calculate how much free volume is required for the hotel guests
total_free_vol = N_guests * vol; % (ft^3)

% Determine how many habitation modules are required to house all the
% guests (rounding up)
N_habs = ceil(total_free_vol / hab_free_volume);

% Here we determine the mass of each hotel component
bulkhead_thickness = .4; % (in)
rho = .103; % Al 2219 (lb/in^3)

% Habitation modules
hab_dia     = 170; % (in)
hab_length  = 500; % (in)
node_dia    = 96;  % Nodes at the ends of each habitation module (in)

hab_mass = rho * hab_length * pi * (((hab_dia/2)^2) - ...
           (((hab_dia - 2*bulkhead_thickness)/2)^2));

% Here the mass for welds (1%) and internal structure (10%) are added
hab_mass = hab_mass + (hab_mass * .01) + (hab_mass * .1);

% Here the Whipple Shield mass is determined (radius is 4.2" larger than
% module)
whipple_density     = .0975; % Al 6061 (lb/in^3)
whipple_thk         = .08; % (in)
whipple_mass_hab    = whipple_density * hab_length * pi * ...
                      ((((hab_dia/2) + 4.2)^2) - ...
                      ((((hab_dia - 2*whipple_thk)/2) + 4.2)^2));

% Final habitation module mass plus 1.5 factor since calculations are very
% rough
hab_mass = (hab_mass + whipple_mass_hab)* 1.5; % (lb)

% Determine dimensions of overall hotel structure
% Assume that the circumference of the habitation module ring is roughly
% equivalent to the sum of the lengths of the habitation modules
hotel_dia = (N_habs * (hab_length + node_dia)) / pi; % (in)

% Interconnecting cylidrical structural elements
ic_dia      = 48; % (in)

% Assume the lengths of the interconnecting tubes is roughly equivalent to
% the radius of the hotel ring
ic_length   = hotel_dia / 2; % (in)

ic_mass = rho * ic_length * pi * (((ic_dia/2)^2) - ...
           (((ic_dia - 2*bulkhead_thickness)/2)^2));

% Here the mass for welds (1%) and internal structure (10%) are added
ic_mass = (ic_mass * .01) + (ic_mass * .1);

% Here we determine the Whipple Shield mass for the IC modules
whipple_mass_ic = whipple_density * ic_length * pi * ...
                  ((((ic_dia/2) + 4.2)^2) - ...
                  ((((ic_dia - 2*whipple_thk)/2) + 4.2)^2));

% Final habitation module mass plus 1.5 factor since calculations are very
% rough
ic_mass = (ic_mass + whipple_mass_ic)* 1.5;

% Here we assume the center cylinder mass is equivalent to that of a habitation
% module
center_module_mass = hab_mass; % (lb)

% Here the total structural mass is calculated (lb)
str_mass = (hab_mass * N_habs) + (ic_mass * N_habs) + center_module_mass;

% Convert to kilograms from pounds
str_mass = str_mass * .454;  % (kg)

% Determine spin rate required to produce artificial gravity in habitation
% modules
g = 32.2; % (ft/s^2)

spin_rate = sqrt(g * (hotel_dia/2)); % (ft/s)

% Convert to meters/sec

spin_rate = spin_rate * .3048;  % (m/s)