function [radius, inclination,comm_power,diameter,solar_array_size,battery_size,power_mass] = Scenario(data_amt, latitude, frequency, diameter_grnd_antenna, daylight_power_needed, eclipse_power_needed, mission_lifetime)
    data_rate = 96000; % bps from SMAD
    epsilon_min = 5 * pi/180; % radians
    
    time =  compute_communication_time(data_amt, data_rate);
    [radius, inclination, max_distance] = compute_feasible_circular_LEO(time, epsilon_min, latitude);
    tic
    
    for i=1:length(radius)
        [comm_power(i), diameter(i)] = comm_sys(data_rate, max_distance(i), frequency, diameter_grnd_antenna);
        periods = calculatePeriods(radius(i), inclination(i));
        solar_array_size(i,:) = calculatePSA(daylight_power_needed, eclipse_power_needed, mission_lifetime, periods);
        battery_size(i,:) = size_batteries(eclipse_power_needed, periods);
        
        for j=1:length(solar_array_size(i,:)) %solar array loop
            for k = 1:length(battery_size(i,:))
                power_mass(i,2*(j - 1) + k) = solar_array_size(i,j) + battery_size(i,k);
            end
        end
        toc
    end
    
    
function [r, i_max, D_max] = compute_feasible_circular_LEO(T_min, epsilon_min, lat_gs)
% [r, i_max, D_max] = compute_feasible_circular_LEO(T_min, epsilon_min, lat_gs)
% Input
%   T_min           minimum communication time per fly-by [sec]
%   epsilon_min     minimum satelite elevation from surface [rad]
%   lat_gs          latitude of the ground station [rad]
% Output
%   r               radius of the orbit [m]
%   i_max           maximum inclination [rad]
%   D_max           maximum distance to ground station [m]
mu_E  = 398600.4418e9;  % Earth gravitational constant [m^3/s^2]
R_E = 6378136.49;       % Earth equatorial radius [m]
omega_E = 7.292115e-5;  % Earth angular velocity [rad/s]
r_max = R_E + 5000000;  % Maximum raidus for which this module is valid [m]
n = 50;               % Number of data points

% Compute the minimum radius for which the ground station is in view for at least T_min.
r_min = fsolve(@min_radius_function, 100000000000, optimset, epsilon_min, lat_gs, T_min);
% If the minimum radius is greater than the maximum radius, then this
% module is no longer valid for the problem.
if (r_min > r_max | abs(imag(r_min)) > 1)
    error('No solution can be found for the given problem using this module!');
else
    r_min = real(r_min);
end

% Generate a set of radius
if (n <= 1)
    r = r_min;
else
    r = r_min:(r_max-r_min)/ceil(n-1):r_max;
end

rho = asin(R_E./r);  % Earth Angular Radius
eta_max = asin(sin(rho)*cos(epsilon_min));  % Maximum nadir angle [rad]
lambda_max = pi/2 - epsilon_min - eta_max;  % Maximum Earth central angle [rad]
P = 2*pi*sqrt(r.^3/mu_E);  % Period of the orbit
lambda_min = acos(cos(lambda_max)./cos(T_min*pi./P));  % Worst case minimum Earth central angle [rad]
lat_pole_max = lat_gs - lambda_min;  % Minimum latitude of the instantaneous orbit pole [rad]
i_max = lat_pole_max;  % Maximum orbit inclination
D_max = R_E*sin(lambda_max)./sin(eta_max);  % Maximum distance to the ground station [m]

%figure(3)
%plot(r/R_E,sin(lambda_max),r/R_E,sin(eta_max),r/R_E,sin(lambda_max)/sin(eta_max))

function x = min_radius_function(r, epsilon_min, lat_gs, T)
mu_E  = 398600.4418e9;  % Earth gravitational constant [m^3/s^2]
R_E = 6378136.49;       % Earth equatorial radius [m]

rho = asin(R_E./r);  % Earth Angular Radius
eta_max = asin(sin(rho)*cos(epsilon_min));  % Maximum nadir angle [rad]
lambda_max = pi/2 - epsilon_min - eta_max;  % Maximum Earth central angle [rad]
P = 2*pi*sqrt(r.^3/mu_E);  % Period of the orbit
lat_pole_min = 0;  % Minimum latitude of the instantaneous orbit pole [rad]
lambda_min_max = lat_gs - lat_pole_min;  % Worst case minimum Earth central angle [rad]
x = P/pi.*acos(cos(lambda_max)./cos(lambda_min_max)) - T;  % The computed communication time should equal the required communication time.

function T_max = compute_communication_time(D_max, R_min)
% T_max = compute_communication_time(D_max,R_min)
%   This module assume communication initiation time of 2 min and adds in a
%   margin of 3 to the quantity of data.
%
%   Ref: Wertz and Larson. Space Misson Analysis and Design, 2nd ed.
%
% Input
%   D_max           Maximum quantity of Data [bit]
%   R_min           Minimum data transfer rate [bit/sec]
% Output
%   T_max           Maximum required communication time [sec]
T_initiate = 2*60; % Communication initation time [sec]
M = 3; % Margin to account for missed passes
T_max = (D_max*M/R_min + T_initiate); % From SMAD 3rd ed.

function [output_data] = calculatePSA(dpn, epn, lifetime, periods)
    Xe_direct_energy_transfer = 0.65;  % Efficiency during eclipse for direct energy transfer
    Xd_direct_energy_transfer = 0.85;  % Efficiency during daylight for direct energy transfer
    Xe_peak_power_tracking = 0.6;  % Efficiency during eclipse for peak power tracking
    Xd_peak_power_tracking = 0.8;  % Efficiency during daylight for peak power tracking
    Id = 0.77;  % Nominal value for inherent degredation
    theta = 0.4101;  % The solar array is at worstcase Sun angle between equatorial and ecliptic planes
    material_degradation_GA = .0275;  % Gallium Arsenide degrades at 2.75% per year (worst case)
    material_degradation_multijunction = .005;  % Multijunction Solar cells degrade at 0.5% per year (worst case) 
    material_degradation_Si = .0375;  % Silicon degrades at 2.75% per year (worst case)
    
%     [periods] = calculatePeriods(altitude, inclination);
    [periods] = [periods] / 60;  % Convert seconds to minutes
            
    psa_det = ((epn * periods(1)) / Xe_direct_energy_transfer + (dpn * periods(2)) / Xd_direct_energy_transfer) / periods(2);
    psa_ppt = ((epn * periods(1)) / Xe_peak_power_tracking + (dpn * periods(2)) / Xd_peak_power_tracking) / periods(2);
    
    power_output_Si = 202.316;  % 14.8% * 1,367 W/m^2 (incident solar radiation)
    power_BOL_Si = powerBeginningLife(power_output_Si, Id, theta);
    power_EOL_Si = powerEndLife(power_BOL_Si, material_degradation_Si, lifetime);
    
    power_output_multijunction = 300.74;  % 22% * 1,367 W/m^2 (incident solar radiation)
    power_BOL_multijunction = powerBeginningLife(power_output_multijunction, Id, theta);
    power_EOL_multijunction = powerEndLife(power_BOL_multijunction, material_degradation_multijunction, lifetime);
    
    power_output_GA = 252.895;  % 18.5% * 1,367 W/m^2 (incident solar radiation)
    power_BOL_GA = powerBeginningLife(power_output_GA, Id, theta);
    power_EOL_GA = powerEndLife(power_BOL_GA, material_degradation_GA, lifetime);
    
    silicon_area_direct_energy_transfer = psa_det / power_EOL_Si;
    multijunction_area_direct_energy_transfer = psa_det / power_EOL_multijunction;
    gallium_arsenide_area_direct_energy_transfer = psa_det / power_EOL_GA;

    silicon_area_peak_power_tracking = psa_ppt / power_EOL_Si;
    multijunction_area_peak_power_tracking = psa_ppt / power_EOL_multijunction;
    gallium_arsenide_area_peak_power_tracking = psa_ppt / power_EOL_GA;
    
    output_data(1) = silicon_area_direct_energy_transfer * 0.55; % 0.55 denisty of silicon cells
    output_data(2) = multijunction_area_direct_energy_transfer * 0.85; % 0.85 denisty of multijunction cells
    output_data(3) = gallium_arsenide_area_direct_energy_transfer * 0.85; % 0.85 denisty of ga-arsenide cells;
    output_data(4) = silicon_area_peak_power_tracking * 0.55; % 0.55 denisty of silicon cells;
    output_data(5) = multijunction_area_peak_power_tracking * 0.85; % 0.85 denisty of multijunction cells;
    output_data(6) = gallium_arsenide_area_peak_power_tracking * 0.85; % 0.85 denisty of ga-arsenide cells;
    
% determines the beginning of life power production
% theta - Sun incidence angle between the vector normal to the surface in degrees
% output - power at beginning of life (W/m^2)
function power_BOL = powerBeginningLife(power_output, inherent_degradation, theta)
    power_BOL = power_output * inherent_degradation * cos(theta);
   
% determines the end of life power production
% output - power at end of life (W/m^2)
function power_EOL = powerEndLife(power_BOL, material_degradation, lifetime) %lifetime in years
    power_EOL = power_BOL * ( (1 - material_degradation) ^ lifetime );

function [battery_mass] = size_batteries(epn, periods)
    NiH2_dod = 0.40; % Worst case from SMAD
    NiCd_dod = 0.10; % Worst case from SMAD
    N = 1;
    n = 0.9;
%     [periods] = calculatePeriods(altitude, inclination);
    [periods] = [periods] / 60; % Convert seconds to minutes
    NiH2_capacity = (epn * periods(1)) / (NiH2_dod * N * n);
    NiCd_capacity = (epn * periods(1)) / (NiCd_dod * N * n);
    NiH2_mass = NiH2_capacity / 35;
    NiCd_mass = NiCd_capacity / 45;
    battery_mass(1) = NiH2_mass;
    battery_mass(2) = NiCd_mass;
    
function [periods] = calculatePeriods(radius, inclination)
    stkinit;
    remMachine = stkDefaultHost;
    conid = stkOpen(remMachine);  % Open the Connect to STK
   
    % first check to see if a scenario is open
    % if there is, close it
    scen_open = stkValidScen;
    if scen_open == 1
        stkUnload('/*')
    end

    cmd = 'New / Scenario maneuver_scenario';  % set up scenario
    stkExec(conid, cmd);
    cmd = 'New / */Satellite sat1';  % put the satellite in the scenario
    stkExec(conid, cmd);

    % set the scenario epoch
    epochDate = '"28 Sep 2003 00:00:00.00"';
    startDate = epochDate;
    stopDate = '"2 Oct 2003 00:00:00.00"';
    cmd = ['SetEpoch * ' epochDate];
    stkExec(conid, cmd);
    stkSyncEpoch;
    
    % set the time period for the scenario
    stkSetTimePeriod(startDate, stopDate, 'GREGUTC');
    
    % set the animation parameters
    rtn = stkConnect(conid,'Animate','Scenario/maneuver_scenario','SetValues "28 Sep 2003 00:00:00.0" 60 0.1');
    rtn = stkConnect(conid,'Animate','Scenario/maneuver_scenario','Reset');

    % set up initial state
    % STK expects fields in meters NOT kilometers
    cmd = ['SetState */Satellite/sat1 Classical J2Perturbation ' startDate ' ' stopDate ' 60 J2000 ' epochDate ' ' num2str(radius) ' 0 ' num2str(inclination*180/pi) ' 0 0 0'];
    stkExec(conid, cmd);
    
    % get eclipse duration from STK
    [secData, secNames] = stkReport('*/Satellite/sat1', 'Eclipse Times');
    if (length(secData{1})==0)
        eclipse_duration = 0;
        
		% set eclipse duration in seconds
		eclipse_duration_average = 0;
		
		% get sunlight duration from STK
		[secData, secNames] = stkReport('*/Satellite/sat1', 'Sun');
		sunlight_duration = stkFindData(secData{1}, 'Duration');
				
		% return eclipse and sunlight periods in seconds
		periods(1) = 0;
		periods(2) = sunlight_duration;

    else
        eclipse_duration = stkFindData(secData{1}, 'Total Duration');
        
        % set eclipse duration in seconds
        eclipse_duration = unique(eclipse_duration);
        x = length(eclipse_duration);
        eclipse_duration = eclipse_duration(2 : (x-1));
        eclipse_duration_average = mean(eclipse_duration);
        
        % get sunlight duration from STK
        [secData, secNames] = stkReport('*/Satellite/sat1', 'Sun');
        sunlight_duration = stkFindData(secData{1}, 'Duration');
        
        % set sunlight duration in seconds
        y = length(sunlight_duration);
        sunlight_duration = sunlight_duration(2 : (y-1));
        sunlight_duration_average = mean(sunlight_duration);
        
        % return eclipse and sunlight periods in seconds
        periods(1) = eclipse_duration_average;
        periods(2) = sunlight_duration_average;
    end
        
    stkClose(conid)  % close out the stk connection
    stkClose  % this closes any default connection
    
function [P, dia_sat] = comm_sys(R, d, freq, dia_gnd)
% Input:
%  - R, data rate, [bits/s]
%  - d, distance from satellite to ground antenna [m]
%  - freq, communication frequency, [Hz]
%  - dia_gnd, ground antenna diameter [m]
% Output:
%  - P, satellite communication required signal power [W]
%  - dia_sat, satellite antenna diameter [m]
% Assumes:
%  - ground antenna at 300 K
%  - satellite antenna gain = 60
%  - communciation frequency < 20 GHz
SNR = 10; % Conservative Signal-to-Noise ratio, SMAD p. 551.
eff_gnd = 0.5; % Ground antenna efficiency
eff_sat = 0.5; % Satellite antenna efficiency
G_sat = 60; % Satellite antenna gain
L_a = 100; % Approximate atmosphere attenuation (rain, clouds, etc.)

N0 = 1.38e-23 * 300; % Noise Density, (Boltzmann's constant [J/K]) * (Ground Antenna Temp [K])
L = (4*pi*d*freq/3e8)^2; % Free space signal loss
dia_sat = sqrt(G_sat*3e8^2/(eff_sat*pi^2*freq^2)); % receiver diameter, [m]
G_gnd = eff_gnd * (pi*dia_gnd*freq/3e8)^2; % Ground antenna gain
P = ( SNR * N0 * R * L * L_a ) / (G_sat * G_gnd); % Satellite communication required signal power, [W]