function [r,i_max,D_max] = compute_feasible_circular_LEO(T_min,epsilon_min,lat_gs)
% [r,i_max,D_max] = compute_feasible_circular_LEO(T_min,epsilon_min,lat_gs)
%
% Input
%   T_min           minimum communication time per fly-by [sec]
%   epsilon_min     minimum satelite elevation from surface [rad]
%   lat_gs          latitude of the ground station [rad]
% Output
%   r               radius of the orbit [m]
%   i_max           maximum inclination [rad]
%   D_max           maximum distance to ground station [m]

mu_E  = 398600.4418e9;  % Earth gravitational constant [m^3/s^2]
R_E = 6378136.49;       % Earth equatorial radius [m]
omega_E = 7.292115e-5;  % Earth angular velocity [rad/s]
r_max = R_E + 5000000;  % Maximum raidus for which this module is valid [m]
n = 10;               % Number of data points

% Compute the minimum radius for which the ground station is in view for at
% least T_min.
r_min = fsolve(@min_radius_function,100000000000,optimset,epsilon_min,lat_gs,T_min);
% If the minimum radius is greater than the maximum radius, then this
% module is no longer valid for the problem.
if (r_min > r_max | abs(imag(r_min)) > 1)
    error('No solution can be found for the given problem using this module!');
else
    r_min = real(r_min);
end

% Generate a set of radius
if (n <= 1)
    r = r_min;
else
    r = r_min:(r_max-r_min)/ceil(n-1):r_max;
end

% Earth Angular Radius
rho = asin(R_E./r);

% Maximum nadir angle [rad]
eta_max = asin(sin(rho)*cos(epsilon_min));

% Maximum Earth central angle [rad]
lambda_max = pi/2 - epsilon_min - eta_max;

% Period of the orbit
P = 2*pi*sqrt(r.^3/mu_E);

% Worst case minimum Earth central angle [rad]
lambda_min = acos(cos(lambda_max)./cos(T_min*pi./P));

% Minimum latitude of the instantaneous orbit pole [rad]
lat_pole_max = lat_gs - lambda_min;

% Maximum orbit inclination
i_max = abs(lat_pole_max);

% Maximum distance to the ground station [m]
D_max = R_E*sin(lambda_max)./sin(eta_max);
%figure(3)
%plot(r/R_E,sin(lambda_max),r/R_E,sin(eta_max),r/R_E,sin(lambda_max)/sin(eta_max))


function x = min_radius_function(r,epsilon_min,lat_gs,T)

mu_E  = 398600.4418e9;  % Earth gravitational constant [m^3/s^2]
R_E = 6378136.49;       % Earth equatorial radius [m]

% Earth Angular Radius
rho = asin(R_E./r);

% Maximum nadir angle [rad]
eta_max = asin(sin(rho)*cos(epsilon_min));

% Maximum Earth central angle [rad]
lambda_max = pi/2 - epsilon_min - eta_max;

% Period of the orbit
P = 2*pi*sqrt(r.^3/mu_E);

% Minimum latitude of the instantaneous orbit pole [rad]
lat_pole_min = 0;

% Worst case minimum Earth central angle [rad]
lambda_min_max = lat_gs - lat_pole_min;

% The computed communication time should equal the required communication
% time.
x = P/pi.*acos(cos(lambda_max)./cos(lambda_min_max)) - T;


