function [m,w,l,I_xx,I_yy] = compute_truss_properties(baseline)
% baseline

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Structure Requirement
minimum_fundamental_frequency = 5*2*pi; % [rad/s]

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Material Properties
modulus_of_elesticity = 220e9;          % [N/m^2] Graphite/Epoxy
mass_density = 1640;                    % [kg/m^3] Graphite/Epoxy

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Design Assumption
member_radius = 0.025;                   % [m]
member_shell_thickness = 0.002;          % [m]
member_radius_inner = member_radius - member_shell_thickness;

% Assume the the mass per length, rho, is approximately equal to the mass of a
% single bin per bin length. A single bin consists of 4 longerons, 4
% diagnonal members, and 4 cross-sectiona members. A bin is assumed to be a
% cube.
mass_per_length = (8+4*sqrt(2))*mass_density*pi*...
                  (member_radius^2 - member_radius_inner^2);    % [kg/m]

% Beam Bending: E*I*d^4(y)/(dx)4 - rho*omega^2*y = 0
%   where E is modulus of elesticity, I is the area moment of intertia, y
%   is the vertical displacement at the beam lenght x, rho is mass per unit
%   length, and omega is the vibartional frequency. The solution of the
%   equation is:
%
%       y = A*cosh(beta*x) + B*sinh(beta*x) + C*cos(beta*x) + D*sin(beta*x)
%
%   Assuming pin connections at each end of the beam, the boundary
%   condition is:
%
%       y(0) = 0, y'(0) = 0, y(l) = 0, y'(l) = 0
%
%   where l is the lenght of the beam. This results in the solution:
%
%       cosh(beta*l)*cos(beta*l) = 1
%
%   beta*l that corresponds to the fundamental mode is:
%
%       beta*l_fundamental = 4.73004074486271
%
%   The natural frequency is:
%
%       omega_n = (beta*l)^2*sqrt(E*I/rho/l^4)
%
%   Then, given the minimum fundamental vibration frequency, the minimum
%   area moment of inertia can be computed. This assumes that the tortional
%   vibration is negligible.
%
%       I = (omega_n/(beta*l)^2)^2*rho*l^4/E
fundamental_beta_l = 4.73004074486271;
minimum_area_moment_of_inertia = (minimum_fundamental_frequency/fundamental_beta_l^2)^2*...
                                 mass_per_length*baseline.^4/modulus_of_elesticity;
                             
% Assuming that the diagonal members do not contribute to beam bending, the
% area moment of inertia is:
%
%   I = pi*(r_o^4 - r_i^4 + (r_o^2 - r_i^2)*l^2)
%
% This assumption is valid since the bending deflection is assumed small
% and the diagonal members are in the middle whwere it goes under both
% compression as well as tension.
minimum_bay_length = sqrt((minimum_area_moment_of_inertia/pi - ...
                           member_radius^4 + ...
                           member_radius_inner^4)...
                          /(member_radius^2 - member_radius_inner^2));
minimum_bay_number = ceil(baseline./minimum_bay_length);
minimum_truss_length = minimum_bay_number.*minimum_bay_length;
minimum_truss_mass = mass_per_length.*minimum_truss_length + ...
                     4*mass_density*pi*(member_radius^2 - member_radius_inner^2).*minimum_bay_length;

% Assuming that the truss is much llike a retangular prism shell
retangular_prism_inner_volume = (minimum_bay_length - 4*member_radius).^2.*minimum_truss_length;
retangular_prism_outer_volume = minimum_bay_length.^2.*minimum_truss_length;
retangular_prism_shell_volume = retangular_prism_outer_volume - retangular_prism_inner_volume;
mass_density_of_retangular_prism = minimum_truss_mass./retangular_prism_shell_volume;
retangular_prism_inner_mass = mass_density_of_retangular_prism.*retangular_prism_inner_volume;
retangular_prism_outer_mass = mass_density_of_retangular_prism.*retangular_prism_outer_volume;
mass_moment_of_inertia_xx = 1/12*retangular_prism_outer_mass.*(2*minimum_bay_length.^2) - ...
                            1/12*retangular_prism_inner_mass.*(2*(minimum_bay_length - 4*member_radius).^2);
mass_moment_of_inertia_yy = 1/12*retangular_prism_outer_mass.*(minimum_bay_length.^2 + minimum_truss_length.^2) - ...
                            1/12*retangular_prism_inner_mass.*((minimum_bay_length - 4*member_radius).^2 + minimum_truss_length.^2);

m = minimum_truss_mass;
w = minimum_bay_length;
l = minimum_truss_length;
I_xx = mass_moment_of_inertia_xx;
I_yy = mass_moment_of_inertia_yy;

stiffness = (minimum_fundamental_frequency./fundamental_beta_l^2).^2*mass_per_length.*l.^4;
figure;
plot(stiffness);