function [Delta_V,Delta_t] = compute_Hohmann_transfer(mu,r_A,r_B)
% [DELTA_V,DELTA_T] = COMPUTE_HOHMANN_TRANSFER(MU,R_A,R_B)
%
% Inputs:
%   MU              Gravitational constant of the body orbiting
%   R_A             Radius of the inner circular orbit
%   R_B             Radius of the outer circular orbit
%
% Outputs:
%   DELTA_V         Total Delta_V necessary for Hohmann transfer
%   Delta_T         Hohman transfer time

% reject any invalid inputs
if (r_A > r_B)
    error('Error! Inner orbit radius is larger than the outer.');
end

v_iA = sqrt(mu/r_A);                % Velocity for inner orbit [m/s]
v_fB = sqrt(mu/r_B);                % Velocity for outer orbit [m/s]
a_tx = (r_A + r_B)/2;               % Semi-major axis of the transfer orbit [m/s]
v_txA = sqrt(mu*(2/r_A - 1/a_tx));  % Velocity at periapsis for the transfer orbit [m/s]
v_txB = sqrt(mu*(2/r_B - 1/a_tx));  % Velocity at apoapsis for the transfer orbit [m/s]
Delta_V_A = v_txA - v_iA;           % Delta_V at periapsis of the transfer orbit [m/s]
Delta_V_B = v_fB - v_txB;           % Delta_V at apoapsis of the transfer orbit [m/s]
Delta_V = Delta_V_A + Delta_V_B;    % Total Delta V required for Hohmann transfer [m/s]
Delta_t = pi*sqrt(a_tx^3/mu);       % Hohmann transfer time [s]
