function [Delta_V,Delta_t] = compute_high_energy_transfer(mu,r_A,r_B,Delta_V_max)
% [DELTA_V,DELTA_T] = COMPUTE_HIGH_ENERGY_TRANSFER(MU,R_A,R_B,Delta_V_max)
%
% Inputs:
%   MU              Gravitational constant of the body orbiting
%   R_A             Radius of the inner circular orbit
%   R_B             Radius of the outer circular orbit
%   Delta_V_max     Maximum impulsive Delta_V available
%
% Outputs:
%   DELTA_V         Total Delta_V necessary for Hohmann transfer
%   Delta_T         Orbit transfer time

% Constrain so that the first burn is no greater than the max allowed.
Delta_V_A = Delta_V_max;

v_iA = sqrt(mu/r_A);                % Velocity for inner circular orbit [m/s]
v_fB = sqrt(mu/r_B);                % Velocity for outer circular orbit [m/s]

% Compute the transfer orbit assuming that the maximum Delta_V_max is used
% at point A.
v_txA = Delta_V_max + v_iA;         % Velocity for transfer orbit at first burn [m/s]
a_tx = 1/(2/r_A - v_txA^2/mu);      % Semi-major axis of the transfer orbit [m]
e = 1 - r_A/a_tx;                   % Eccentricity of the transfer orbit
nu = acos((a_tx*(1-e^2)/r_B - 1)/e);% True anomaly at the second burn [rad]
phi = atan(e*sin(nu)/(1+e*cos(nu)));% Flight path angle at second burn [rad]
v_txB = sqrt(mu*(2/r_B - 1/a_tx));  % Velocity for transfer orbit at second burn [m/s]
% Second burn Delta_V necessary [m/s]
Delta_V_B = sqrt(v_fB^2 + v_txB^2 - 2*v_fB*v_txB*cos(phi));

% Check to see if the second burn is less than the maximum allowed.
if (Delta_V_B <= Delta_V_max)
    % The first burn is the limiting factor.
    Delta_V = Delta_V_A + Delta_V_B;            % Total Delta_V [m/s]
    E = acos((e + cos(nu))/(1+e*cos(nu)));      % Eccentricy anomaly at second burn [rad]
    Delta_t = sqrt(a_tx^3/mu)*(E-e*sin(E));     % Orbit transfer time [sec]
    return
else
    % The second burn is the liminting factor.
    Delta_V_B = Delta_V_max;                    % Delta_V at second burn [m/s]
    a_tx = search_a_tx(mu,r_A,r_B,Delta_V_B);   % Semi-major axis of transfer oribt [m]
    v_txA = sqrt(mu*(2/r_A - 1/a_tx));          % Velocity for tranfer orbit at first burn [m/s]
    Delta_V_A = v_txA -v_iA;                    % Delta_V at first burn [m/s]
    Delta_V = Delta_V_A + Delta_V_B;            % Total Delta_V [m/s]
    e = 1 - r_A/a_tx;                           % Eccentricity of the transfer orbit
    nu = acos((a_tx*(1-e^2)/r_B - 1)/e);        % True anomaly at the second burn [rad]
    E = acos((e + cos(nu))/(1+e*cos(nu)));      % Eccentricy anomaly at second burn [rad]
    Delta_t = sqrt(a_tx^3/mu)*(E-e*sin(E));     % Orbit transfer time [sec]
end

function Delta_V_B = compute_Delta_V_B(a_tx,mu,r_A,r_B)
e = 1 - r_A/a_tx                                % Eccentricity of the transfer orbit
nu = acos((a_tx*(1-e^2)/r_B - 1)/e);            % True anomaly at the second burn [rad]
phi = atan(e*sin(nu)/(1+e*cos(nu)));            % Flight path angle at second burn [rad]
v_txB = sqrt(mu*(2/r_B - 1/a_tx));              % Velocity for transfer orbit at second burn [m/s]
v_fB = sqrt(mu/r_B);                            % Velocity for outer circular orbit [m/s]
Delta_V_B = sqrt(v_fB^2 + v_txB^2 - 2*v_fB*v_txB*cos(phi))

function a_tx = search_a_tx(mu,r_A,r_B,Delta_V_B)
allowable_percent_error = 0.000001;
a_tx_low = r_B/2;
a_tx = (r_A + r_B)/2;
a_tx_high = (r_A + r_B)*10000;
quit = 0;
while (not(quit))
    difference = Delta_V_B - compute_Delta_V_B(a_tx,mu,r_A,r_B); % This should be zero
    if ((a_tx_high - a_tx)/a_tx < allowable_percent_error | ...
            (a_tx - a_tx_low)/a_tx < allowable_percenT_error);
        quit = 1;
    elseif (difference > 0)
        a_tx_low = a_tx;
        a_tx = (a_tx_high - a_tx)/2;
    elseif (difference < 0)
        a_tx_high = a_tx;
        a_tx = (a_tx - a_tx_low)/2;
    end    
end
