function [t,L,A,E,W,life] = MER_sol4
%[T, L, A, E, W, LIFE] = MER_sol4
%
% Reads in the solar array input power and the load power for the Mars Exploration Rover
% egress scenario (data are from the MER Mission Plan, JPL/NASA).  Requires the Excel
% spreadsheet 'MER_sol4_input_data.xls' to be in the same directory.  Determines the 
% profile of the battery state of charge given the load and source power profiles, 
% and plots the results.
%
% Outputs:
%   T       [min] Time vector
%   L       [W] Power load profile
%   A       [W] Power source profile
%   E       Time profile of fraction of battery being used
%   W       [W] Power dissipation profile
%   LIFE    Fraction of remaining usable battery lifetime
%

% read in data from the Excel spreadsheet
[data,clocks] = xlsread('MER_sol4_input_data.xls');

% time history of the load power
val    = data(:,1);
val    = val(find(val(:,1)>=0));
times  = clocks(:,1);
times  = times(1:length(val));
time   = t2ms(times);
[tL,L] = create_profile(time, val, 4);

% time history of the array power
val    = data(:,3);
val    = val(find(val(:,1)>=0));
times  = clocks(:,3);
times  = times(1:length(val));
time   = t2ms(times);
[tA,A] = create_profile(time, val, 0);

if (sum(tA-tL)~=0)
    error('Error!  Load and input power profiles have different time vectors.');
end

t     = tA;                 % common time profile
dt    = 60;                 % [s] chosen to simplify calculations
life0 = 1.0;                % fraction of life span remaining at start of period
energy_max     = 448*3600;  % Joules
energy_initial = 0.59*energy_max;

% battery properties
efficiency     = 1.0;
slope          = -0.28;
intercept      = 1.67;

% find the output
[EE, invalid, W, life] = battery_profile(length(t), dt, L, A, energy_initial, ...
                         energy_max, life0, efficiency, slope, intercept);

% warn if energy drops below zero
if length(invalid)>0
    disp('Battery energy went negative; invalid design.');
end

% change to a percentage
E = EE/energy_max;

% plot the data
h=figure(1);
subplot(311)
plot_results('original', t/60, L, A, E, W);
subplot(312)
plot_results('ourdata', t/60, L, A, E, W);
subplot(313)
plot_results('all', t/60, L, A, E, W);

% scale the plot
hmargin = 0;
vmargin = 0;
orient tall;
set(h,'PaperPosition',[hmargin vmargin (8.5-2*hmargin) (11-2*vmargin)]);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Plots the time profiles of L, A, E, and W as functions of t.
% 'What' tells which data are to be plotted.
function plot_results(what, t, L, A, E, W)

if strcmp(what,'original') | strcmp(what,'all')
    ax(1) = newplot;
    set(gcf,'nextplot','add')
    
    % read in and display the MER power profile image
    img = imread('MER_sol4_power_profile.png','png');
    h1  = image(img);
    set(ax(1),'visible','off');
    set(ax(1),'box','off')
    
    % rescale the width by 80% (left-justified)
    pos    = get(ax(1),'position');
    pos(3) = 0.8*pos(3);
    set(ax(1),'position',pos);
end

if strcmp(what,'original')
    % create a dummy axis on which to make a legend
    dummyax = axes('position',[1 1 1/0.8 1].*get(ax(1),'position'));

    % make dummy plots with the colors we will use for the real data
    plot(-1,-1,'b-',-1,-1,'g-',-1,-1,'m-',-1,-1,'r-');
    axis([0 1 0 1]);
    set(dummyax,'color','none','box','off','XTick',[],...
        'XTickLabel',[],'YTick',[],'YTickLabel',[])

    % create the legend
    h=legend('Load power','Source power','Excess power','Battery SOC',-1);
    title('Energy balance for MER A egress (Sol 4) from MER Mission Plan');
    return;
end

if strcmp(what,'ourdata')
    ax(2)  = newplot;
    pos    = get(ax(2),'position');
    pos(3) = 0.7*pos(3);
    set(ax(2),'position',pos);
else
    ax(2) = axes('position',get(ax(1),'position'));
end

if strcmp(what,'ourdata') | strcmp(what,'all')
    % the source, load and excess powers, and a line to make the top look nice
    h2 = plot(t, A, 'g-', t, L, 'b-', t, W, 'm-', [0 24],[140 140],'k-');
    if strcmp(what,'ourdata')
        set(ax(2),'xgrid','off','ygrid','off','box','off');
    else
        set(ax(2),'color','none','xgrid','off','ygrid','off','box','off');
    end
    set(gcf,'nextplot','add')
    axis([0 24 0 140]);
    set(ax(2),'XTick',[]);
    ylabel('Power [W]')
    
    % the battery state of charge percentage, with right-side axis
    ax(3) = axes('position',get(ax(2),'position'));
    h2 = plot(t, 100*E, 'r-');
    set(ax(3),'YAxisLocation','right','color','none', ...
        'xgrid','off','ygrid','off','box','off');
    axis([0 24 0 100]);
    set(ax(3),'XTick',[0 6 12 18 24]);
    ylabel('Battery state of charge [%]')
end

set(gcf,'nextplot','replace');
xlabel('Time [Mars hours]');

if strcmp(what,'ourdata')
    title('Simulation data.  Outputs are battery SOC and excess power.');
else
    title('Simulation data overlayed on MER mission plan data');
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% translates corner value hh:mm to minutes
function corner_times = t2ms(times)
for i=1:length(times)
    corner_times(i) = t2m(times{i});
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% creates a power profile given corner values.  corner_times must be integers.
function [t,L] = create_profile(corner_times, corner_values, initial)

% check for validity of inputs
if length(corner_times) ~= length(corner_values)
    error('Error!  Invalid input to create_profile()');
end
if sum(corner_times==round(corner_times))~=length(corner_times)
    error('Error!  Input to create_profile() ''corner_times'' must contain only integers.');
end

% make sure there is a time zero
t = 0:corner_times(length(corner_times));
if (corner_times(1) ~= 0)
    corner_times = [0 corner_times];
    corner_values = [initial corner_values];
end

% fill in the time profile with equal steps
for i=1:length(corner_times)-1
    t0 = corner_times(i);
    t1 = corner_times(i+1);
    dt = t1-t0;
    L(t0+2:t1+1) = (corner_values(i+1)-corner_values(i))/dt*(1:dt)+corner_values(i);
end
L(1) = initial;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% changes hh:mm to number of minutes since midnight
function [min] = t2m(time)
div = find(time==':');
if length(div)~=1
    error('Error!  Invalid input to function t2m().');
end
min = str2num(time(1:div-1))*60 + str2num(time(div+1:length(time)));


