function R = PowerDesignResult(power_source, energy_storage, power_load, environment, t);
% R = POWERDESIGNRESULTS(POWER_SOURCE, ENERGY_STORAGE, POWER_LOAD, ENVIRONMENT, T)
%
% Inputs:
%  POWER_SOURCE             Design specification of the power source. 
%  ENERGY_STORAGE           Energy storage device design specification.
%  POWER_LOAD               Power load requirement.
%  ENVIRONMENT              Environment specification.
%  T                        Time.
%
% Outputs:
%  R                        Struct of the results:
%   .Mass                   Total power subsystem mass [Kg].
%   .Cost                   Total cost of the power subsystem [US$].
%   .AvailablePower         Power availble from the power source [W].
%   .AvailableEnergy        Energy available from the energy storage [J].
%   .EnergyStorageInvalid   Time at which the enery was insufficient.
%   .EnergyDissipation      Required energy dissipation.
%   .EnergyStorageLifeRemaining     Remaining life on the energy storage.

% Initialize the mass of the design.
R.Mass = 0;
% Initialize the cost of the design.
R.Cost = 0;
% Initialize the available power.
R.AvailablePower = zeros(1,length(t));
% Initialize the available engergy.
R.AvailableEnergy = zeros(1,length(t));




% For each power source:
for i = 1:length(power_source)
    % The power source is a solar array:
    if strcmp(power_source{i}.Device,'Solar Array')
        %if not(isfield(enviornment,'illumination_intensity'))
        %    error('To use a solar array, the illumination intensity "illuminationIntensity" must be specified within the environment")
        %end
        
        % Compute the effective illumination intensity given the
        % illumination intensity and the incident angle of of the
        % illumination.
        effective_intensity = ...
            EffectivePowerIntensity(environment.IlluminationIntensity, ...
                                    power_source{i}.IncidentAngle);
        % Compute the power generated pwer area of the solar array.
        power_per_area = SolarPowerPerArea(power_source{i},effective_intensity,t);
        % Compute the power available.
        R.AvailablePower = R.AvailablePower ...
                         + power_source{i}.SurfaceArea * power_per_area;
        % Compute the mass of the solar array.
        R.Mass = R.Mass ...
               + power_source{i}.Density * power_source{i}.SurfaceArea;
           R.SAMass=power_source{i}.Density * power_source{i}.SurfaceArea;
           R.RTGMass=0;
           R.MassINV=0;
        R.CostINV=0;
        % Compute the cost of the solar array.
        R.Cost = R.Cost ...
               + power_source{i}.CostPerArea * power_source{i}.SurfaceArea;
    % The power source is an RTG:
    elseif strcmp(power_source{i}.Device,'RTG')
        % Compute the power availble.
        R.AvailablePower = R.AvailablePower + RTGPower(power_source{i},t);
        % Lookup the mass of the RTG.
        R.Mass = R.Mass + power_source{i}.Mass;
        R.MassINV=0;
        R.CostINV=0;
        R.SAMass=0;
        R.RTGMass=power_source{i}.Mass;
        % Lookup the cost of the RTG.
        R.Cost = R.Cost + power_source{i}.Cost;
    % Do not recognize the power source type:
    else
        error('Unrecognized power source was defined.');
    end
end

[slope, intercept] = slope_intercept(energy_storage.DOD, energy_storage.Cyclelife);

% Compute energy available from the energy storage device.
[R.AvailableEnergy, R.EnergyStorageInvalid, ...
 R.EnergyDissipation, R.EnergyStorageLifeRemaining] = battery_profile(...
        length(t), ...                              % Number of time steps
        t(2) - t(1), ...                            % Time increment
        power_load, ...
        R.AvailablePower, ... 
       ...%energy_storage.InitialCharge, ...           % Initial energy available
       energy_storage.SpecificEnergy * energy_storage.Mass * 0.6, ...    % Initial energy available                          
      energy_storage.SpecificEnergy * energy_storage.Mass, ...    % Max chargin capacity                          
    energy_storage.RemainingLife, ...           % Remaing Life
        energy_storage.CoulombicEfficiency, ...     % Efficiency
        slope, ...
        intercept);
if (length(R.EnergyStorageInvalid) > 0)
    %disp('Invalid energy storage device.')
   R.MassINV=R.Mass + energy_storage.Mass;
R.CostINV=R.Cost + energy_storage.Cost;

R.Mass=NaN;
R.Cost=NaN;
R.STMass=NaN;

else
    
% Compute the mass if the energy storage device.
R.Mass = R.Mass + energy_storage.Mass;
R.STMass=energy_storage.Mass;
% Cost of the power subsystem design.
R.Cost = R.Cost + energy_storage.Cost;
R.MassINV=NaN;
R.CostINV=NaN;
end
