function [energy, invalid, dissipation, life] = battery_profile(nsteps, dt, load, source, e_init, e_max, life_init, efficiency, slope, intercept)
%[ENERGY, INVALID, DISSIPATION, LIFE] 
%     = BATTERY_PROFILE(NSTEPS,DT,LOAD,SOURCE,E_INIT,E_MAX,LIFE_INIT,EFFICIENCY,SLOPE,INTERCEPT)
%
% Inputs:
%   NSTEPS      The number of uniformly-spaced time values
%   DT          [s] The separation between the time values
%   LOAD        [W] The time profile of the power load
%   SOURCE      [W] The time profile of the power source
%   E_INIT      [J] The initial stored energy in the storage device
%   E_MAX       [J] The capacity of the energy storage device
%   LIFE_INIT   The fraction of the device's usable life as of the start of the period
%   EFFICIENCY  The fraction of the input energy that is recoverable
%   SLOPE       The slope of the depth of discharge vs log life fraction plot
%   INTERCEPT   The y-intercept of the depth of discharge vs log life fraction plot
%
% Outputs:
%   ENERGY      [J] The time profile of the energy stored in the device
%   INVALID     Indices of times at which the stored energy is negative (!)
%   DISSIPATION [W] The time profile of excess energy to be dissipated
%   LIFE        The fraction of the device's usable life as of the end of the period
%
% Note that if LENGTH(INVALID)>0, the capacity of the energy storage device is insufficient
% to meet demand, and this is an invalid design.  Any function calling BATTERY_PROFILE(...)
% should verify that INVALID is empty before proceeding with further calculations.
%

% reject any invalid inputs
if (nsteps<2)
    error('Error!  NSTEPS must be at least two.');
end
if (dt<=0)
    error('Error!  Time step DT must be greater than zero.');
end
if (length(load) ~= nsteps)
    error('Error!  Length of LOAD must be NSTEPS.');
end
if (length(source) ~= nsteps)
    error('Error!  Length of SOURCE must be NSTEPS.');
end
if (e_max<0)
    error('Error!  Energy capacity E_MAX must be non-negative.');
end
if (e_init>e_max)
    error('Error!  Initial energy E_INIT is greater than max energy E_MAX.');
end
if (life_init > 1 | life_init < 0)
    error('Error!  Initial life LIFE_INIT must be between zero and one.');
end
if (efficiency > 1 | efficiency < 0)
    error('Error!  Charge efficiency EFFICIENCY must be between zero and one.');
end

% energy starts with e_init
energy(1) = e_init;

% invalid vector starts out with no indices
invalid = [];

% actual capacity is about 95% of e_max, based on MER battery profile
e_cap = 0.95*e_max;

% life begins at initial value
life = life_init;

% create a time index vector
time = 0:length(load)-1;

% allocate memory now to speed things up
dissipation = zeros(size(time));

% integrate power and track energy storage profile
for i=2:length(time)
    energy(i) = energy(i-1) + efficiency*dt*(source(i-1)-load(i-1));
    
    % energy can't go negative, so this should result in an invalid design
    if energy(i)<0
        % records the indices of invalid energy events
        invalid = [invalid i];
    end

    if energy(i)>e_cap
        % records the excess energy that won't fit in the storage device
        dissipation(i) = (energy(i)-e_cap)/dt;
        
        % caps the stored energy at its maximum value
        energy(i) = e_cap;
    end
end

% all the indices that have negative slope
dec = find(sign(energy(2:length(energy))-energy(1:length(energy)-1)) == -1);

% all the indices that have positive slope
inc = find(sign(energy(2:length(energy))-energy(1:length(energy)-1)) ==  1);

% account for depth of discharge and cycles to track life expectancy
i=1;
while i<length(dec)
    % the first point in a descent (i.e. the local peak)
    top  = max([(dec(i)-1) 1]);
    
    % the indices in the next increasing section
    temp = inc(find(inc>top));
    
    % if there are no more increasing sections
    if length(temp)==0
        % depth of discharge
        dod = (e_max-energy(dec(length(dec))))/e_max;
        
        % terminate the loop
        i=length(dec);
        
    % if there is an upcoming increasing section
    else
        % interested in the first index in the increasing section
        temp = temp(1);
        
        % depth of discharge
        dod = (e_max-energy(temp))/e_max;
        
        % find the indices falling in later decreasing sections
        dec = dec(find(dec>temp));
    end
    
    % decrement the lifetime by some fraction due to this cycle
    life = life - delta_life(dod, slope, intercept);
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Determines the fraction of the storage device lifetime lost 
% due to a particular discharge cycle.
function dlife = delta_life(dod_frac, m, b)
if m==0 & b==0
    dlife = 0;
else
    dlife = 10^-((dod_frac-b)/m);
end
