function plot_battery_dod_cycles(varargin)
% PLOT_BATTERY_DOD_CYCLES
% PLOT_BATTERY_DOD_CYCLES(FIGNUM)
% PLOT_BATTERY_DOD_CYCLES(AXES_HANDLE)
%
% Plots the depth of discharge (DOD) as a function of cycle repetition lifetime
% Creates a new figure, or plots to figure number FIGNUM.

% read the data from the Excel spreadsheet
[storage, generation, solar] = power_read_xls('power_design_vector.xls');

% check if any data were found
slen = length(storage);
if slen==0
    disp('Warning.  No energy storage device data found.');
    return;
end

% check to see where it is supposed to plot
if nargin>0
    var = varargin{1};
    if var-fix(var) == 0
        % set the figure
        figure(varargin{1});
    else
        % set the axes
        axes(var)
    end
else
    % create a new figure
    figure;
end

% record hold state
washold = ishold;

% set up plotting properties for easy access later
colors = 'bgrmcy';
colors = [colors colors colors colors colors colors colors];
points = 'ox+sd<>';
points = [points points points points points points points];

% initialize
types{1} = '';
legstr = [];
count = 0;

% look at all elements in the storage structure
for i=1:slen
    % check to see if we have seen this type yet
    if ~ismember(storage(i).Type, types)
        % add the newly found type to the list of known types
        count = count+1;
        types{count+1} = storage(i).Type;
        
        % get the line properties from the battery characteristics
        [slope, intercept] = slope_intercept(storage(i).DOD, storage(i).Cyclelife);
        
        % ignore the case when fatigue is not a factor
        if slope==0 & intercept==0
            continue;
            
        else
            % add on the the string that we will eval for the legend
            legstr = [legstr ',''' storage(i).Type ' (' storage(i).PartNumber ')' ''''];
            
            % offset the points so we can see overlapping data sets
            x=[-1:7] + count/8;
            y=slope*x+intercept;
            
            % plot the line
            plot(x,100*y,[colors(count) points(count) '-']);
        end
        hold on;
    end
end

% return the hold state to its previous value
if ~washold
    hold off;
end

% set nice axis limits and turn on the grid
axis([0 8 0 100]);
grid on;

% negatize the x ticks, since we're looking at the inverse of the plotted value
h=gca;
ticks = str2num(get(h,'XTickLabel'));
set(h,'XTickLabel',num2str(-ticks));

% label the plot
title('Depth of discharge vs cycle life for secondary batteries');
xlabel('Fraction of life lost in one charge/discharge cycle [log_{10}(x)]');
ylabel('Depth of discharge [%]');

% create a legend
legstr = legstr(2:length(legstr));
eval(['legend(' legstr ')']);

