% Matthew Richards
% Anna Silbovitz
% Optimizing power
function [cell_type, M_sa, area] = Power_Optimizing(L_m, P_av, P_dl, T_dl, T_e, T_d, inc)
% This function selects the optimal solar cell type as a function of area

% INPUTS
% L_m: mission life (years)
% P_av: power requirement, not including data transmission needs (W)
% P_dl: power required for data tranmission needs (W)
% T_dl: time to downlink data (minutes)
% T_e: average time of eclipse (minutes)
% T_d: average time not in eclipse (minutes)
% inc: incliation of the orbit (rad)

% OUTPUTS
% cell_type: name of solar cell
% M_sa: mass of solar array (kg)
% area: area of solar array (m^2)

% ASSUMPTIONS
% ***

disp('Finding Solar Array...');

% Defining problem variables (source: SMAD)
X_e = .65;                          %energy transfer efficiency during eclipse, direct energy transfer
X_d = .85;                          %energy transfer efficiency during daylight, direct energy transfer
S_i = 1367;                         %solar illumination intensity, W/m^2
X_si = .148;                        %solar cell production efficiency, Silicon 
X_th = .050;                        %solar cell production efficiency, Thin Sheet Amorphous Si
X_ga = .185;                        %solar cell production efficiency, Gallium Arsenide
X_ip = .180;                        %solar cell production efficiency, Indium Phosphide
X_mj = .220;                        %solar cell production efficiency, Multijunction GaInP/GaAs
I_d = .77;                          %inherent degradation
theta = 23.5 * (pi/180) + inc;        %worst-case sun angle, radians

% Compute power collected by solar array during daylight
P_sa = (P_av * T_e / X_e + P_av * T_d / X_d + P_dl * T_dl) / T_d;

% Vector of solar cell efficiencies
X = [X_si X_th X_ga X_ip X_mj];

% Compute power ouptut for various solar cell types (W/m^2)
P_o = S_i .* [X];

% Compute beginning-of-life power (W) per unit area for various solar cell types
P_bol = (cos(theta) * I_d) .* P_o;

% Compute actual lifetime degradation for various solar cell types
A_d = [.0375 .0375 .0275 .0375 .0375];
L_d = (1 - A_d) .^ L_m;

% Compute end-of-life power (W) per unit area for various solar cell types
P_eol = P_bol .* L_d;

% Compute solar array area (m^2) required to support input power requirement
A_sa = P_sa ./ P_eol;


% List best solar cell type (smallest cell area)
fprintf('\n');
disp('Best solar cell type, minimizing for area:');
Min = min(A_sa);
if Min == A_sa(1);
    cell_type = 'Silicon'
elseif Min == A_sa(2);
    cell_type = 'Thin Sheet Amorphous Si'
elseif Min == A_sa(3);
    cell_type = 'Gallium Arsenide'
elseif Min == A_sa(4);
    cell_type = 'Indium Phosphide'
elseif Min == A_sa(5);
    cell_type = 'Multijuction GaInP/GaAs'
end

% Compute mass (kg) of solar array
M_sa = .04 * P_sa;

% Area of chosen solar array
area = Min;

disp('Solar array mass (kg): ');
disp(M_sa);
disp('Solar array area (m^2): ');
disp(area);







