% Storage Optimizing
% Matthew Richards
% Anna Silbovitz

function [battery_name, mass] = Storage_Optmizing(L_m, cyc, P_av, P_dl, T_dl, T_e)
% This function selects the optimal battery as a function of mass

% INPUTS
% L_m: life of mission (years)
% cyc: cycle life (number of times satellite goes in and out of sun)
% P_av: average power requirement, not including for data transmission (W)
% P_dl: power for data transmission (W)
% T_dl: time of each data transmission (minutes)
% T_e: time of eclipse (minutes)

% OUTPUTS
% battery_name
% mass: mass of battery (kg)


disp('Finding Battery...');

% Defining problem variables (source: SMAD)
X_nc = 30;          %Nickel-Cadium specefic energy (W*hr/kg)
X_nhi = 43;         %Nickel-Hydrogen (individual pressure) specefic energy (W*hr/kg)
X_nhc = 56;         %Nickel-Hydrogen (common pressure) specefic energy (W*hr/kg)
X_nhs = 57;         %Nickel-Hydrogen (single pressure) specefic energy (W*hr/kg)
X_li = 110;         %Lithium-Ion specefic energy (W*hr/kg)
X_ss = 210;         %Sodium-Sulfur specefic energy (W*hr/kg)
n = 0.90;            %Transmission effeciency between the battery and the load

% Vector of specific energy densities
X = [X_nc X_nhi X_nhc X_nhs];

% Determine Depth-of-Discharge
if cyc < 1000
    DoD_nc = .65;
    DoD_nhi = .90;
    DoD_nhc = .90;
    DoD_nhs = .90;
elseif cyc >= 1000 & cyc < 2000
    DoD_nc = .55;
    DoD_nhi = .75;
    DoD_nhc = .75;
    DoD_nhs = .75;
elseif cyc >= 2000 & cyc < 4000
    DoD_nc = .45;
    DoD_nhi = .65;
    DoD_nhc = .65;
    DoD_nhs = .65;
elseif cyc >= 4000 & cyc < 10000
    DoD_nc = .35;
    DoD_nhi = .55;
    DoD_nhc = .55;
    DoD_nhs = .55;
elseif cyc >= 10000 & cyc < 20000
    DoD_nc = .30;
    DoD_nhi = .50;
    DoD_nhc = .50;
    DoD_nhs = .50;
elseif cyc >= 20000 & cyc < 40000
    DoD_nc = .20;
    DoD_nhi = .40;
    DoD_nhc = .40;
    DoD_nhs = .40;
elseif cyc > 40000
    DoD_nc = .15;
    DoD_nhi = .35;
    DoD_nhc = .35;
    DoD_nhs = .35;
end

% Determine Power Required During Eclipse
% Design for worst-case: assume downlink during eclipse
P_e = (P_av * T_e + P_dl * T_dl) / T_e;

% Determine Battery Capacity (W*hr)
DoD = [DoD_nc DoD_nhi DoD_nhc DoD_nhs];
C_r = (P_e * T_e/60) .* ((n * DoD).^-1);

% Determine Mass of Storage System
M = C_r ./ X;

% Display Lightest Power Storage System
fprintf('\n');
disp('Best secondary battery type (lightest):');
Min = min(M);
if Min == M(1);
    battery_name = 'Nickel-Cadium'
elseif Min == M(2);
    battery_name = 'Nickel-Hydrogen (individual pressure vessel)'
elseif Min == M(3);
    battery_name = 'Nickel-Hydrogen (common pressure vessel)'
elseif Min == M(4);
    battery_name = 'Nickel-Hydrogen (single pressure vessel)'
end

% Display Mass of Lightest Power Storage System
disp('Battery Mass (kg):');
Min = min(M);
disp(Min)
mass = Min;
