function [period, orbit] = find_orbit(site_lat, site_lon, alt_p, gst)

% finds the orbital elements for an orbit that repeats once a sidereal day,
% and has it's radius of perigee over the ground site

% INPUTS
% site_lat: latitude of the ground site, in degrees
% site_lon: longitude of the ground site, in degrees
% alt_p: altitude of perigee of the orbit, in kilometers
% gst: start time as Greenwich sidereal time
 
% OUTPUTS
% orbit: orbital elements [semi-major axis (km), eccentricity, inclination
% (deg), argument of perigee (deg), longitude of ascending node (deg)
% period: orbit period in minutes


% ASSUMPTIONS
% 1. the earth is perfectly circular, with radius = 6378.137 km
% 2. site elevation isn't significant enough to use
% 3. eccentricity is flexible, but is set close to zero, but in a way that
    % satisfies requirements for repeating ground track orbit
% 4. inclination can be in a range; it will be fixed to be 5 degrees above the
    % value of the site latitude; if site latitude is over 85 degrees,
    % inclination will be 90

    

disp('Finding Orbit...')

% radius of the earth
r_earth = 6378.137;

% constant used for finding period: 2*pi/mu^.5
period_const = 0.00016587;

% length of sidereal day in minutes
sid_day = 1436.068167;

% radius of perigee
rp = r_earth + alt_p;

% start with e being nearly circular
% find period with this value, then adjust period so that it equals 
% (1 sidereal day)/(integer number of revolutions)
% then, readjust e and a

e_initial = 0.01;
a_initial = rp/(1.0 - e_initial);
P_initial = period_const*a_initial^(3/2);

% we want number of revs to be an integer
n_revs = sid_day/P_initial;
% using floor to raise n_revs makes period go up slightly; so semi-major
% axis will go up slightly, radius of apogee will stay larger than radius
% of perigee, and eccentricity will stay positive
int_revs = floor(n_revs);

% period, semi-major axis, eccentricity
P = sid_day/int_revs;
a = (P/period_const)^(2/3);
e = 1.0 - rp/a;

% inclination
if(abs(site_lat) > 85)
    i = 90;
else
    i = abs(site_lat) + 5;
end

% argument of perigee to put perigee over the ground site
w = asin(sin(site_lat*pi/180)/sin(i*pi/180))*180/pi;
while(w > 360)
    w = w - 360;
end
while(w < 0)
    w = w + 360;
end

% longitude of ascending node
% first find target position's longitude meridian
b = asin(cos(i*pi/180)/cos(site_lat*pi/180));
lm = acos(cos(b)/sin(i*pi/180));

l_an = gst + site_lon - lm*180/pi;
while(l_an > 360)
    l_an = l_an - 360;
end
while(l_an < 0)
    l_an = l_an + 360;
end

orbit = [a, e, i, w, l_an];
period = P;

disp('semi-major axis (km): ');
disp(a);
disp('eccentricty: ');
disp(e);
disp('inclination (deg): ')
disp(i);
disp('argument of perigee (deg): ');
disp(w);
disp('longitude of ascending node (deg): ')
disp(l_an);






