function [data_sent, orbit, M_sa, A_sa, M_bat] = ground_comm(site_lat, site_lon, pass_alt, dr, pow, start_date, life)

% top level function
% this function takes in user inputs, and calls functions to find orbit,
% amount of data sent and power for transmitter, and power system

% INPUTS
% site_lat: latitude of the ground site, in degrees
% site_lon: (east) longitude of the ground site, in degrees
% pass_alt: height above the ground the satellite should pass when passing
            % directly over the ground site, in kilometers
% dr: data rate of data transmitted from spacecraft to groundsite, in bps
% pow: power needs of the spacecraft, not including for data transmission,
            % in watts
% start_date: start time of mission, which is time satellite is first at
        % perigee over ground site. format is: 
        % [year, month, day, hour, min]
% life: mission length, in years

% OUTPUTS
% data_sent: total amount of data sent during each pass (average), in bits
% orbit: orbital elements [semi-major axis (km), eccentricity, inclination
% (deg), argument of perigee (deg), longitude of ascending node (deg)
% M_sa: mass of solar array
% A_sa: area of solar array
% M_bat: mass of battery

if(pass_alt > 3000)
    disp('Altitude at perigee must be < 3000 km');
    return;
end



% first find gst of date
yr = start_date(1);
mo = start_date(2);
d = start_date(3);
h = start_date(4);
min = start_date(5);
gst = find_gst(yr, mo, d, h, min);

% then find orbit based on site, altitude requirement
[period, orbit] = find_orbit(site_lat, site_lon, pass_alt, gst);

% set up orbit in STK, to find eclipse times and site passage times
[eclipse, pass_time] = run_stk(orbit, site_lat, site_lon, start_date);

% fnd power needs for transmission, amount of data sent each time
[data_sent, pow_dl] = transmitter(dr, pass_time);

% find solar array
[cell_type, M_sa, A_sa] = Power_Optimizing(life, pow, pow_dl, pass_time, eclipse, (period-eclipse), orbit(3)*pi/180);


% find battery
cyc = life*365*24*60/period;
[battery_name, M_bat] = Storage_Optimizing(life, cyc, pow, pow_dl, pass_time, eclipse);

