function [eclipse, pass_time] = run_stk(orbit, site_lat, site_lon, date)

% opens stk, sets up the orbit and the ground site
% runs scenario 
% finds average time craft is in eclipse each orbit
% finds average time of each pass over ground site (average of closest pass
% each day)

% INPUTS
% orbit: [semi-major axis (km), eccentricity, i (degrees), w (degrees), ascending node (degrees)]
% site_lat: site latitude in degrees
% site_lon: site longitude in degrees
% date: start date of orbit (time of first perigee) 
        % [year, month, day, hour, minute]

% OUTPUTS
% eclipse: average time each orbit craft is in eclipse, in minutes
% pass_time: average time of the longest pass each day, in minutes

% ASSUMPTIONS
% Averaging the needed values over the length of the mission takes WAY too
% long in STK.  Because of the repeating orbit and the cyclic nature of
% space, we assume it's adequate to average over a 10 day period
% The scenario is set up for 12 days, with the first and the last removed,
% so as not to include partial passes
% we realize to get a real average for eclipse time, we'd probably have to
% average over a year; however for the purposes of this project, the 10 day
% average is assumed to be close enough


disp('Connecting to STK...');

year = date(1);
day = date(3);
hour = date(4);
min = date(5);

% initialize connection to STK, if this has not yet been done
stkinit;

% use the default connection data for connecting to STK
% port should be 5001, hostname 'localhost': this setup is what STK
% recommends and guides you to do when STK and Matlab are on the same
% machine
remMachine = stkDefaultHost;

%open the connection to STK
conid=stkOpen(remMachine);

% first check to see if a scenario is open
% if there is, close it
scen_open = stkValidScen;

if scen_open == 1
    stkUnload('/*')
end

% set up scenario
cmd = 'New / Scenario ground_comm';
stkExec(conid, cmd);

% put the satellite in the scenario
cmd = 'New / */Satellite sat1';
stkExec(conid, cmd);


% put the ground site in the scenario
cmd = 'New / */Facility ground_site';
stkExec(conid, cmd);

if(date(2) == 1)
    month = 'Jan';
elseif(date(2) == 2)
    month = 'Feb';
elseif(date(2) == 3)
    month = 'Mar';
elseif(date(2) == 4)
    month = 'Apr';
elseif(date(2) == 5)
    month = 'May';
elseif(date(2) == 6)
    month = 'June';
elseif(date(2) == 7)
    month = 'July';
elseif(date(2) == 8)
    month = 'Aug';
elseif(date(2) == 9)
    month = 'Sept';
elseif(date(2) == 10)
    month = 'Oct';
elseif(date(2) == 11)
    month = 'Nov';
elseif(date(2) == 12)
    month = 'Dec';
else
    disp('invalid date, month must be 1 through 12');
end
startDate = ['"', num2str(day), ' ', month, ' ', num2str(year), ' ', num2str(hour), ':', num2str(min), ':00.0"'];
stopDay = day + 12;
stopDate = ['"', num2str(stopDay), ' ', month ' ', num2str(year), ' ', num2str(hour), ':', num2str(min) ':00.0"'];
epochDate = startDate;


% set the scenario epoch
cmd = 'SetUnits / km sec GregUTC';
stkExec(conid, cmd);
cmd = ['SetEpoch * ' epochDate];
stkExec(conid, cmd);
stkSyncEpoch;
% set the time period for the scenario
stkSetTimePeriod(startDate, stopDate, 'GREGUTC');
% set the animation parameters
tmp = ['SetValues ', startDate, ' 60 0.1'];
rtn = stkConnect(conid,'Animate','Scenario/ground_comm',tmp);
rtn = stkConnect(conid,'Animate','Scenario/ground_comm','Reset');


% set orbit elements
cmd = ['SetState */Satellite/sat1 Classical TwoBody ' startDate, ' ', stopDate, ' 60 J2000 ', epochDate, ' ', num2str(orbit(1)), ' ', ...
        num2str(orbit(2)), ' ', num2str(orbit(3)), ' ', num2str(orbit(4)), ' ', num2str(orbit(5)), ' 0.0'];
stkExec(conid, cmd);

% set ground site position
cmd = ['SetPosition */Facility/ground_site Geocentric ', num2str(site_lat), ' ', num2str(site_lon), ' 0.0'];
stkExec(conid, cmd);

% get access data, satellite to ground site
intervals = stkAccess('*/Satellite/sat1', '*/Facility/ground_site');
[n,m] = size(intervals);

% loop through access data, eliminating first partial day and last partial day,
% find the pass overhead - at perigee, the repeating pass once a sidereal
% day - that we want to use

% length of sidereal day in minutes
sid_day = 1436.068167;

perigee_pass = zeros(1,10);
for(j=1:10)
   for(i=1:n)
       start_min = intervals(i).start/60;
       stop_min = intervals(i).stop/60;
       if(start_min < j*sid_day && stop_min > j*sid_day)
           perigee_pass(1,j) = stop_min - start_min;
       end
   end
end

% get average pass length
pass_time = mean(perigee_pass);


% find average eclipse time
[data, names] = stkReport('*/Satellite/sat1', 'Eclipse Times');
all_times = stkFindData(data{1}, 'Total Duration');
ecl_numbers = stkFindData(data{1}, 'Start Pass Number');
% for some reason ecl_numbers is not numeric, and won't convert right as an
% array, so convert one at a time
for(i=1:length(ecl_numbers))
    ecl_n(i) = str2num(ecl_numbers(i,:));
end
% the eclipse report lists several lines of data for each eclipse; the
% total duration values are repeated, but we don't want duplicates - find
% uniques; i are the indices we want
[b, i, j] = unique(ecl_n);
unique_times = all_times(i);
% remove first and last times, in case they are partial eclipse values
% average the rest
len = length(unique_times);
eclipse = mean(unique_times(2:len-1))/60;


stkClose(conid);

