function [actuator1, actuator2, torque1, momentum1, torque2, momentum2] = ACSsize(a, e, MU, Cd, A, rt, moment_arm, D, Imax, Imin, pointing_accu, pointing_req, slew_dist, slew_time)

% function  ACSsize.m calculates the size of ACS actuators given mission
% objectives and spacecraft characteristics
% a  [   m   ]: semimajor axis of the orbit
% e  [       ]: eccentricity of the orbit
% MU [m^3/s^2]: gravitational parameter for the central body
% Cd [   ]: the coefficient of drag on the spacecraft
% A  [m^2]: the "shadow area" of the spacecraft (worst-case cross-sectional area)
% rt [ m ]: the moment arm for the drag force (center of pressure - center
% of mass)
% moment_arm = [ m ] - moment arm from thruster to cg
% D [amp*m^s]: residal dipole of the vehicle
% Imax, Imin [kg*m^2]: maximum and minimum moments of inertia of the spacecraft
% pointing_accu [degrees]: pointing accuracy required for mission
% pointing_req [  ]: where you want your spacecraft pointing (ep == earth-pointing,
% ip == inertial pointing)
% slew_dist [degrees] - amount needed to slew
% slew_time [seconds] - amount of time needed to slew



[r,v] = ae2rv(a, e, MU);

%find disturbance torques

tq_aero = t_aero(r, Cd, A, v, rt);

tq_bfield = t_bfield(r, D);

tq_g = t_gg(r, Imax, Imin, MU);

tq_solar = t_solar(A, rt);

% map mission objectives to type of actuators
[actuator1, actuator2] = obj2actuator(pointing_accu, pointing_req, slew_dist/slew_time);

%determine sizes

%assume a circular orbit for calculating periods, and I_max for use in
%craft inertia
P = (2*pi*r)/(60*v);
torque1 = 0;
torque2 = 0;
momentum1 = 0;
momentum2 = 0;
if(strcmp(actuator1, 'rw') == 1 || strcmp(actuator1, 'cmg') == 1)    
    [torque1, momentum1] = reacwheel_cmgsize(tq_aero, tq_bfield, tq_g, tq_solar, slew_dist, slew_time, Imax, P);
elseif(strcmp(actuator1, 'mw') == 1)
    momentum1 = momwheelsizing(tq_aero, tq_bfield, tq_g, tq_solar, pointing_accu, P);
elseif(strcmp(actuator1, 'thruster') == 1)
    torque1 = thrustersizing(tq_aero, tq_bfield, tq_g, tq_solar, slew_dist, slew_time, moment_arm, I_max);
else
    torque1 = magtorsizing(tq_aero,tq_bfield,tq_g,tq_solar);
end

if(strcmp(actuator2, 'rw') == 1 || strcmp(actuator2, 'cmg') == 1)
    [torque2, momentum2] = reacwheel_cmgsize(tq_aero, tq_bfield, tq_g, tq_solar, slew_dist, slew_time, Imax, P);
elseif(strcmp(actuator2, 'mw') == 1)
    momentum2 = momwheelsizing(tq_aero, tq_bfield, tq_g, tq_solar, pointing_accu, P);
elseif(strcmp(actuator2, 'thruster') == 1)
    torque2 = thrustersizing(tq_aero, tq_bfield, tq_g, tq_solar, slew_dist, slew_time, moment_arm, Imax);
else
    torque2 = magtorsizing(tq_aero,tq_bfield,tq_g,tq_solar);
end