close all;
clear;
clc;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Generate Transfer Orbits

gravitational_constant_sun =    1.327e20;   %[m^3/s^2], Sun's gravitational constant
distance_sun2earth =            1.49598e11; %[m], average distance from Sun to Earth
distance_sun2mars =             2.28e11;    %[m], average distance from Sun to Mars

% Types of transfer orbits in terms of the semimajor axis of eliptical
% transfer orbit.
semimajor_axis_hohmann =    (distance_sun2earth + distance_sun2mars)/2;
semimajor_axis_max =        2*semimajor_axis_hohmann;
semimajor_axis_increment =  (semimajor_axis_max - semimajor_axis_hohmann)/5;
semimajor_axis =            semimajor_axis_hohmann:semimajor_axis_increment:semimajor_axis_max;

% Compute delta v, total transfer time, and time profile of position and
% true anomaly of the vehicle for each transer orbit type.
for i = 1:length(semimajor_axis);
    [dv1, dv2, T_flight, t, r_mag, nu_time] = ...
        transfer_direct(distance_sun2earth,...
                        distance_sun2mars,...
                        semimajor_axis(i),...
                        gravitational_constant_sun);
    delta_v1(i) = dv1;
    delta_v2(i) = dv2;
    delta_v_total(i) = dv1 + dv2;
    time_total(i) = T_flight;
    time(i,:) = t';
    distance_sun2veh(i,:) = r_mag';
    true_anomaly(i,:) = nu_time';
end

%%%%%%%%%%
% Plot
figure
subplot(2,1,1), ...
	plot(semimajor_axis/distance_sun2earth, delta_v_total/1000, 'ok-', ...
       semimajor_axis/distance_sun2earth, delta_v2/1000, '^b-', ...
       semimajor_axis/distance_sun2earth, delta_v1/1000, 'xr-');
title('Earth-to-Mars Direct Transfer');
xlabel('Transfer Orbit Semi-major Axis [AU]');
ylabel('\DeltaV [km/s]');
legend('total burn', 'final burn', 'initial burn',0);
grid on;
subplot(2,1,2), plot(semimajor_axis/distance_sun2earth, time_total/60/60/24, 'o');
% title('Earth-to-Mars Direct Transfer');
xlabel('Transfer Orbit Semi-major Axis [AU]');
ylabel('Transfer Time [days]');
grid on;

%%%%%%%%%%
% Plot
figure;
color = ['b' 'g' 'r' 'c' 'm' 'y' ...
         'b' 'g' 'r' 'c' 'm' 'y' ...
         'b' 'g' 'r' 'c' 'm' 'y' ...
         'b' 'g' 'r' 'c' 'm' 'y' ...
         'b' 'g' 'r' 'c' 'm' 'y'];
polar([0:pi/20:2*pi],distance_sun2mars/distance_sun2earth*ones(1,41),'k');
hold on;
polar([0:pi/20:2*pi],ones(1,41),'k');
for i = 1:length(semimajor_axis)
    polar(true_anomaly(i,:),distance_sun2veh(i,:)./distance_sun2earth,color(i));
end
hold off;
axis([-2 2 -2 2]);
axis equal;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Life Support System
for i = 1:length(semimajor_axis)
    [power_life_support_day, power_life_support_night, m] = Life_Support(time_total(i), 6, 'iss');
    mass_life_support(i) = m;
end
power_life_support_avg = (16*power_life_support_day + 8*power_life_support_night)/24;
    
figure
plot(time_total/3600/24,mass_life_support,'o-');
xlabel('Transfer Time [days]');
ylabel('Life Support Subsystem Mass [kg]');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Power Subsystem

%take in the design vector
[storage, generation, solar] = power_read_xls('power_design_vector.xls');
storage.RemainingLife = 1.0;
solar.IncidentAngle = 0;

%solar array vector
solar_array_area=50:50:250; %m^2

%battery array vector
battery_mass = 100:100:500; %kg

solar_radiation = 3.826e26; % [W]

% Iterate over all possible power designs
for i = 1:length(semimajor_axis)
    power_load = power_life_support_avg*ones(size(time(i,:)));
    ENV(i).IlluminationIntensity = solar_radiation./(4*pi*distance_sun2veh(i,:).^2);
    for j=1:length(battery_mass)
        storage.Mass = battery_mass(j);
        for k=1:length(solar_array_area)
            solar.SurfaceArea =  solar_array_area(k);
            solar_array_results(i,j,k) = PowerDesignResult({solar}, storage, power_load, ENV(i), time(i,:));
        end
        for k=1:length(generation);
            rtg_results(i,j,k) = PowerDesignResult({generation(k)}, storage, power_load, ENV(i), time(i,:));
        end
    end
end

mass_power = inf*ones(size(semimajor_axis));
mass_battery = inf*ones(size(semimajor_axis));
area_solar_array = inf*ones(size(semimajor_axis));
mass_rtg = inf*ones(size(semimajor_axis));
[i_length,j_length,k_solar_length] = size(solar_array_results);
[i_length,j_length,k_rtg_length]   = size(rtg_results);
for i = 1:i_length
    for j = 1:j_length
        for k = 1:k_solar_length
            if (solar_array_results(i,j,k).Mass < mass_power(i))
                mass_power(i) = solar_array_results(i,j,k).Mass;
                power_subsystem_type{i} = 'solar array';
                area_solar_array(i) = solar_array_area(k);
                mass_rtg(i) = nan;
                mass_battery(i) = battery_mass(j);
            end
        end
        for k = 1:k_rtg_length
            if (rtg_results(i,j,k).Mass < mass_power(i))
                mass_power(i) = rtg_results(i,j,k).Mass;
                power_subsystem_type{i} = 'rtg';
                area_solar_array(i) = nan;
                mass_rtg(i) = generation(k).Mass;
                mass_battery(i) = battery_mass(j);
            end
        end
    end
end

figure
plot(time(1,:)/3600/24,ENV(1).IlluminationIntensity,color(1));
hold on;
for i = 2:length(semimajor_axis)
    plot(time(i,:)/3600/24,ENV(i).IlluminationIntensity,color(i));
end
xlabel('Time of Flight [days]');
ylabel('Solar Intensity [W/m^2]');

figure;
hold on;
for i = 1:i_length
    for j = 1:j_length
        for k = 1:k_solar_length
            plot(time_total/60/60/24,solar_array_results(i,j,k).Mass,'xr');
        end
        for k = 1:k_rtg_length
            plot(time_total/60/60/24,rtg_results(i,j,k).Mass,'*b');
        end
    end
end
xlabel('Transfer Time [days]');
ylabel('Power Subsystem Mass [kg]');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Propulsion Subsystem

% Dry mass is assumed to be structure + life support + power.
mass_dry = mass_life_support + mass_power;
for i = 1:length(semimajor_axis)
    [m_prop, m_sys, m_wet, name] = propulsion_module(delta_v1(i),mass_dry(i));
    mass_propellant{i} = m_prop;
    mass_propulsion_system{i} = m_sys;
    mass_wet{i} = m_wet;
    propulsion_system_type{i} = name;
end

for i = 1:length(time_total)
    for j = 1:length(mass_wet{i})
        mass_wet_new{j}(i) = mass_wet{i}(j);
    end
end

color = ['b' 'g' 'r' 'c' 'm' 'y' 'k'];
shape = ['o' 'x' '+' '*' 's' 'd' '^'];
% figure
% plot(time_total(1)/3600/24*ones(size(mass_wet{1})), mass_wet{1}, [color(1) shape(1) '-']);
% hold on;
% for i = 1:length(time_total)
%     plot(time_total(i)/3600/24*ones(size(mass_wet{i})), mass_wet{i}, [color(i) shape(i) '-']);
% end
% xlabel('Time of Flight [days]');
% ylabel('Total Vehicle Mass [kg]');

figure
plot(time_total/3600/24, mass_wet_new{1}, [color(1) shape(1) '-'],...
     time_total/3600/24, mass_wet_new{2}, [color(2) shape(2) '-'],...
     time_total/3600/24, mass_wet_new{3}, [color(3) shape(3) '-'],...
     time_total/3600/24, mass_wet_new{4}, [color(4) shape(4) '-']);
xlabel('Transfer Time [days]');
ylabel('Total Vehicle Mass [kg]');
legend([propulsion_system_type{1}(1),propulsion_system_type{1}(2),propulsion_system_type{1}(3),propulsion_system_type{1}(4)]);