function [slope, intercept] = slope_intercept(dod, cycles)
%[SLOPE, INTERCEPT] = SLOPE_INTERCEPT(DOD, CYCLES)
%
% Returns the slope and intercept of a plot of the depth of discharge as a function
% of the log10 of the number of cycles in the device lifetime.
%
% Inputs:
%   DOD         Depth of discharge.  Fraction of the energy discharged.
%   CYCLES      Number of cycles possible at given depth of discharge.
%
% Outputs:
%   SLOPE       Slope of plot of DOD as a function of log10 of the cycle life.
%   INTERCEPT   DOD-axis intercept of plot.
%
% If CYCLES=0, SLOPE=0 and INTERCEPT=0 will be returned.  Otherwise, SLOPE=-0.28 
% and INTERCEPT corresponding to the DOD, CYCLES pair will be returned.
%

% check for invalid inputs
if (dod>1 | dod<0)
    error('Error!  Depth of discharge must be between zero and one.');
end
if (cycles<0)
    error('Error!  Number of cycles must be non-negative.');
end

% any device that does not exhibit changes in lifetime with depth of discharge.
if cycles==0
    slope     = 0;
    intercept = 0;
    
% any other device
else
    % from SMAD III, p.421
    slope     = -0.28;
    
    % from y = m * log10(x) + b
    intercept = dod-slope*log10(cycles);
end
