%##############################################
%## Direct Transfers 
%##   (One-Tangent burn ==> Hohmann transfer)
%##
%## Input:
%##  r1 - initial orbit radius [m]
%##  r2 - final orbit radius [m]
%##  a - transfer orbit semi-major axis [m]
%##  mu - central body gravity constant [m^3/s^2]
%##
%## Output:
%##  DeltaV_escape - initial Delta V burn [m/s]
%##  deltaV2 - final Delta V burn [m/s]
%##  T_flight - time of flight [sec]
%##  time - time of flight vector [s]
%##  r_mag - radial position vector over time [m]
%##  nu_time - true anomaly vector over time [rad]
%##
%## Assumes:
%##  - circular initial and final orbits
%##


function [DeltaV_escape, deltaV2, T_flight, time, r_mag, nu_time] = transfer_direct(r1,r2,a,mu)

    e = 1 - r1/a;  %transfer orbit eccentricity

		v1 = sqrt(mu/r1);  %[m/s], initial orbital velocity
		v2 = sqrt(mu/r2);  %[m/s], final orbital velocity

    v1_trans = sqrt(mu*(2/r1 - 1/a));  %[m/s], transfer velocity at initial point
    v2_trans = sqrt(mu*(2/r2 - 1/a));  %[m/s], transfer velocity at final point

    cos_nu = ((a*(1-e^2)/r2-1)/e);  %[rad], true anomaly at final orbit
    nu = acos(cos_nu);  %[rad]
    
    phi = atan(e*sin(nu))/(1+e*cos_nu);  %[rad], flight path angle at final orbit

    deltaV1 = abs(v1_trans - v1);  %[m/s], initial DeltaV from outside earth SOI to 
                                          %begin transfer trajectory.

    deltaV2 = sqrt(v2^2 + v2_trans^2 - 2*v2*v2_trans*cos(phi));  %[m/s], final DeltaV to
                                                       %circularize the orbit once reaching Mars

    T_flight_sun = t_flight(a,e,cos_nu,mu);  %[s], time of flight for sun centered trajectory phase
    
    
    %##############################################
    %## Calculates the orbital transfer trajectory
    
			oe(1) = a; %[m], semi-major axis
			oe(2) = e; %eccentricity
			oe(3) = 0; %[rad], orbit inclination
			oe(4) = 0; %[rad], argument of periapsis
			oe(5) = 0; %[rad], longitude of ascending node
        i = 1;
        
        for nu_current=[0:nu/99:nu]
						oe(6) = nu_current; %[rad], true anomaly (at epoch)
						nu_time(i,1) = nu_current;
						
						%=========================================================
						%= Takes orbital elements of transfer orbit and calculates
						%= the postion for the point current true anomaly.
						
						[r_mag(i,1),v_mag,gamma] = oe2rvg(oe,mu);  %[m], radius
						
						time(i,1) = t_flight(a,e,cos(nu_current),mu);  %[s], time from perigee to nu
						
						i=i+1;
        end
    

    %##############################################
    %## Calculates SOI transfer

        v_inf = deltaV1; %[m/s], v_inf at SOI equals the req DeltaV to initialize the earth/mars transfer
        r_park = 200000+6378000;  %[m], inital earth parking orbit (200km)
        mu_planet = 3.986e14; %[m^3/s^2], earth mu
        rps = r1;  %[m], earth-to-sun distance
	
        [DeltaV_escape, T_flt_escape] = transfer_escape(v_inf, r_park, mu_planet, mu, rps);
        
        T_flight = T_flight_sun + T_flt_escape;  %[s], total time of flight






%##################################################################################
%## Planetary Escape Transfer
%##
%## outputs the deltaV for time from perigee to nu.

function [DeltaV_escape, T_flight] = transfer_escape(v_inf, r_park, mu_planet, mu_sun, rps);

  v_park = sqrt(mu_planet/r_park);  %[m/s], circular velocity at parking
  v1 = sqrt(v_inf^2 + 2*mu_planet/r_park);  %[m/s], desired velocity at parking
  DeltaV_escape = v1-v_park;  %[m/s], escape delta V
  
  E = (v_inf^2)/2;  %energy
  h = r_park*v1;  %angular momentum 
  e_trans = sqrt(1 + 2*E*h^2/mu_planet^2);  %eccentricity of the transfer orbit
  p = 2*r_park;
  r_SOI = rps*(mu_planet/mu_sun)^(2/5);  %[m], sphere of influence radius
  nu = acos((p/r_SOI - 1)/e_trans);  %[rad], true anomaly at SOI
    
  a = 100000000000; %[m], ~inf for hyperbolic orbit
  e = 1 - r_park/a;  %transfer orbit eccentricity

  T_flight = t_flight(a,e,cos(nu),mu_planet);  %[s], time of flight for earth centered trajectory phase
    
    




%##################################################################################
%## time of flight calculation
%##
%## outputs the delta time from perigee to nu.

function [dT] = t_flight(a,e,cos_nu,mu)

  EE = acos( (e+cos_nu) / (1+e*cos_nu) );  %Eccentric anomaly at nu
  dT = sqrt(a^3/mu) * (EE-e*sin(EE));

  
