function [proton_flux,electron_flux] = compute_irradiation_flux...
                                            (year,month,day,hour,minute,second,...
                                             radius,theta,phi,...
                                             proton_energy,electron_energy,solar_max)
% [PROTON_FLUX,ELECTRON_FLUX] =
%   COMPUTE_IRRADIATION_FLUX(YEAR,MONTH,DAY,HOUR,MINUTE,SECOND,
%                            RADIUS,THETA,PHI,PROTON_ENERGY,ELECTRON_ENERGY
%                            , ..
%                            SOLAR_MAX) computes theirradiation flux on an
%                            object.
%
% Input
%   year                UT time year, XXXX [A.D.]
%   month               UT time month, Jan, Feb, ... [1, 2, 3,...]
%   day                 UT time [day]
%   hour                UT time [hr]
%   minute              UT time [min]
%   second              UT time [s]
%   radius              Geocentric Geographic sherical raidus [m]
%   theta               Geocentric Geographic sherical co-latitude [rad]
%   phi                 Geocentric Geographic sherical azmuth [rad]
%   proton_energy       Proton energy of an interest [MeV]
%   electron_energy     Electron energy of an interest [MeV]
%   solar_max           Solar max? true or false [1,0]
%
% Output
%   proton_flux         Omnidirectional proton flux [p/cm^2/s]
%   electron_flux       Omnidirectional electon flux [p/cm^2/s]

R_Earth = 6378137;              % Earth radius [m]

% Decide which data to use.
if (solar_max)
    % If solar max, use AP8MAX and AE8MAX data. 
    AP_data = 'AP8MAX';
    AE_data = 'AE8MAX';
else
    % Otherwise, use AP8MIN and AE8MIN data.
    AP_data = 'AP8MIN';
    AE_data = 'AE8MIN';
end

for i = 1:length(year)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This section uses the International Association of Geomagnetism and Aeronomy
% (IAGA) International Geomagnetic Refernce Field (IGRF) model to compute the
% magnetic field at a given time and location. The supporting functions are
% from:
%           Tsyganenko's GEOPACK Library
%           Translated from original FORTRAN April 10-11, 2003
%           By Paul O'Brien (original by N.A. Tsyganenko)
%           Paul.OBrien@aero.org (Nikolai.Tsyganenko@gsfc.nasa.gov)
%           Translation supported by NSF Grant ATM 0202107
    % Initialize for magnetic field intensity calculation. 
    GEOPACK_RECALC(year,datenum(year(i),month(i),day(i)) - datenum(year(i),1,1) + 1,hour(i),min(i),sec(i));
    % Magnetic field vector at (r,theta,phi) in Geocentric Geographic
    % spherical coordinate [nT].
    [B_r,B_theta,B_phi] = GEOPACK_IGRF_GEO((radius(i)/R_Earth),theta,phi);
    % Magnetic field intensity [nT].
    BetaValueF = sqrt(B_r^2 + B_theta^2 + B_phi^2);
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This section is from the radbelt_trmfun.c code written by:
%     August 10, 1998 Dan Leonard,
%        dleonard@cfa.harvard.edu,
%        (617) 496-7075 
%        (617) 496-7049 fax
% This uses dipole approximation and can be improved. B0 is the magnetic
% field intensity at the magnetic equator. This value should be computed
% using the International Association of Geomagnetism and Aeronomy (IAGA)
% International Geomagnetic Refernce Field (IGRF) model. The L-value
% calculation method should be improved.
    % Compute L value
    bottomF = 4 - (BetaValueF^2*(radius(i)/R_Earth)^6/(3.06e4*3.06e4));
    L       = 3*(radius(i)/R_Earth)/bottomF;

    % From page 8 of the AP-8 Trapped Proton Documnent;
    % also easily derivable from dipole approximation
    B_B0 = BetaValueF*L^3/3.06e4;
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This section uses code translated from trmfun.for code written by:
%       Bieter Bilitza
%       TRMFUN.FOR
%       Trapped Radiation Models Program RADBELT
%       March 25, 1988
%       NASA Goddard Space Flight Center, code 633
    % Compute proton irradiation flux [p/cm^2/s]
    proton_flux(i,:) = 10.^trara1(L,B_B0,proton_energy,AP_data);
    % Compute electron irradiation flux [p/cm^2/s]
    electron_flux(i,:) = 10.^trara1(L,B_B0,electron_energy,AE_data);
end
