%%Function Handles, Anonymous Functions, and ODE45

%[time_array,x_approx] =ode45(@g_Rec8_lesson_hard_coded,[0 t_final],x_initial);
% This is the code to the Matlab ode45 integration, but what does the @
% mean?

%% Function handles
% @ is used to generate function handles. Function handles are means of
% storing function identities as variables within the matlab environment, this allows us
% to identify functions within code without explicitly typing out their name

%Simple Example
h = @sin

%%
% function_handle appears in variable list
whos 

%%
% We can now use h to direct to sin()
h(pi)

%% Passing functions as input arguments
% Function handles are particularily useful in passing functions to other functions
% Note that the function it is passed to must be coded to handle this. Many
% built-in matlab functions are coded to have this capability.
%
% The MATLAB built-in 'quad' integrates a function over given bounds.
% 
help quad

%%
% Note that the first argument of 'quad' is a function handle *not* a function 
% name. That is, 
%
% >> quad(sin,0,pi/2)
%
% will not work. MATLAB will try to evaluate 'sin' and will throw an error 
% because 'sin' needs an input argument. We need to pass in a function handle
% for 'sin'. Fortunately, we already have one:

quad(h,0,pi/2)

%% Anonymous Functions
% We can also define the function at the same time we generate the handle,
% these are known as anonymous functions

square = @(x) x.^2;

a = square(5)


%% Multi-Variable Anonymous Function

sum_of_two_squares = @(x,y) x.^2 + y.^2;

a = sum_of_two_squares(5,5)


%% Workspace variables in anonymous functions
% We can use variables defined in the workspace inside our anonymous functions

a = 1.3;
b = .2;
c = 30;

parabola = @(x) a*x.^2 + b*x + c;

y = parabola(1)

%% 
% These variables are frozen in the function definition

clear a b c

%%

y = parabola(1)

%% 
% If we want to change these variables we must redefine the function

a = -3.9;
b = 52;
c = 0;

y = parabola(1)

%%

parabola = @(x) a*x.^2 + b*x + c;

y = parabola(1)

%% Using function handles to reduce number of inputs
% We can use these concepts to redefine multi-input functions to a lower
% number of inputs

ones(1,5)
ones(2,5)
ones(3,5)

%%
columns=5 

ones_single_input=@(rows) ones(rows,columns)

%%
ones_single_input(1)
ones_single_input(2)
ones_single_input(3)

%% ODE45 demo

%[time_array,x_approx] =ode45(@(t,x) dynamics(t,x,constant),[0,t_final],x_initial);

% We must understand and define each variables within the function call

%% Defining the system of differential equations
%
% First lets choose a function to integrate: 
%
%       m*(d^2q/dt^2) + k*q = A*cos(omega*t)
%
% Mass-spring with an applied sinusoidal force. We can rewrite this as a 
% system of first order ODEs:
%
%   Let x_1 = q, x_2 = dq/dt. Then our system of first-order ODEs is
%
%       dx_1/dt = x_2
%       dx_2/dt = -k/m*x_1 + (A/m)*cos(omega*t)
%
% We need a function which takes in the state-vector, x = [x_1; x_2], and 
% outputs its derivative. The file `dynamics_hard_coded` contains such a
% function for a particular set of values for m, k, and F(t).

t=0;
x=[0 ; 0];

x_dot = dynamics_hard_coded(t,x)

%% Other Inputs for the IVP
% [0,t_final] - Integration time span
% x_initial - Initial conditions
% options - Various settings for solver behaviour

t_final=30;
x_initial= [ 0 ; 1 ]; 
options = odeset('AbsTol',1e-4); % tolerance for error



%% ODE45 demo - Hard Coded

[time_array,x_approx] =ode45(@dynamics_hard_coded,[0 t_final],x_initial,options);

figure(1)
plot(time_array,x_approx(:,1),'b')

title('Displacement vs. Time')
xlabel('Time(s)')
ylabel('Displacement (m)')

figure(2)
plot(time_array,x_approx(:,2),'r')

title('Velocity vs. Time')
xlabel('Time(s)')
ylabel('Velocity (m/s)')


%% ODE45 demo - Changing parameters
% What if we want to simulate the behavior of our mass-spring system for
% different physical parameters (m, k, A, omega)? We can write a dynamics 
% function that takes those parameters as arguments - `dynamics_w_params`
% contains such a function.
%
% However, the ode45 interface specifies that our dynamics function (the one that gives x_dot) must take
% two (and only two) arguments. The first is a time and the second is a
% state. If we want to pass additional arguments to our dynamics function
% we can use an anonymous function:
%
%   @(t,x) dynamics_w_params(t,x,k,A,omega,m)
%
% Here's how that would look:

t_final=30;
x_initial= [ 0 ; 0 ];  
options = odeset('AbsTol',1e-4); % tolerance for error

k=1;
A=1;
omega=1;
m=1;
%These values are for resonance Conditions

[time_array,x_approx] =ode45(@(t,x) dynamics_w_params(t,x,k,A,omega,m),[0 t_final],x_initial,options);

figure(1)
plot(time_array,x_approx(:,1),'b')

title('Displacement vs. Time')
xlabel('Time(s)')
ylabel('Displacement (m)')

figure(2)
plot(time_array,x_approx(:,2),'r')

title('Velocity vs. Time')
xlabel('Time(s)')
ylabel('Velocity (m/s)')


