from random import random
from Line_Groove import *

####################################################################
## ALLELE CODE
####################################################################

class Allele:
    """ base allele class, should never be instantiated """

    def __init__(self):
        raise Exception("base class should not be instantiated.")
    
    def copy():
        """ all alleles must have a copy method """
        raise Exception("not implemented")

    def checkRep(self):
        """ all alleles must have this for debugging purposes """
        raise Exception("not implemented")

class NoteAllele(Allele):
    """ note allele class, holds one midi value """

    def __init__(self, value):
        self.value = value

    def copy(self):
        """ copies this note allele """
        return NoteAllele(self.value)

    def __str__(self):
        return str(self.value)

    def checkRep(self):
        """ debugging purposes """
        if self.value < 0 or self.value > 127: raise Exception("allele out of range exception")

class ChordAllele(Allele):
    """ chord allele class, holds multiple midi values """

    def __init__(self, values):
        self.values = values

    def copy(self):
        """ copies this chord allele """
        newValues = []
        for value in self.values:
            newValues.append(value)
        return ChordAllele(newValues)

    def __str__(self):
        chordStr = "chord allele: "
        for value in self.values:
            chordStr += str(value) +  ", "

        return chordStr[:-2]
    
    def checkRep(self):
        """ debugging purposes """
        for value in self.values:
            if value < 0 or value > 127: raise Exception("allele out of range exception")

class GrainAllele(Allele):
    """ grain allele class, holds multiple attributes that could be applied to granular synthesis """

    def __init__(self, pitch, duration, amplitude):
        self.values = [pitch, duration, amplitude]
        self.PITCH=0
        self.DURATION=1
        self.AMPLITUDE=2

    def copy(self):
        """ copies this grain allele """
        return GrainAllele(self.values[self.PITCH],self.values[self.DURATION],self.values[self.AMPLITUDE])

    def __str__(self):
        return "grain allele: pitch " + str(self.values[self.PITCH]) + ", duration " + str(self.values[self.DURATION]) + ", amplitude " + str(self.values[self.AMPLITUDE])

    def checkRep(self):
        """ debugging purposes """
        for value in self.values:
            if value < 0: raise Exception("allele out of range exception")

####################################################################
## CHROMOSOME CODE
####################################################################

class Chromosome:
    """ base class for chromosomes, holds a list of alleles, should not be instantiated alone """

    def __init__(self, alleles):
        self.alleles = alleles

    def fitness(self, target):
        raise Exception("not implemented")

    def copy(self):
        """ all chromosomes must have a copy method """
        raise Exception("not implemented")

    def mutate(self, other):
        """ all chromosomes must have a mutate method """
        raise Exception("not implemented")

    def checkRep(self):
        """ debugging purposes """
        for allele in self.alleles:
            allele.checkRep()

class NoteChromosome(Chromosome):
    """ basic note chromosome class, holds one note allele """

    def __init__(self, alleles, incrementProbability, maxIncrement):
        Chromosome.__init__(self,alleles)
        self.incProb = incrementProbability # chance of point mutation
        self.maxInc = maxIncrement # maximum point mutation

    def fitness(self, target):
        """ calculates the fitness by comparing to target """
        return 1 - float(abs(target.alleles[0].value - self.alleles[0].value))/127

    def mutate(self, other):
        """ mutates this chromosome """
        if random() > .5: toMutate = self
        else: toMutate = other
        
        numInc = 0
        for i in range(self.maxInc):
            if random() > self.incProb:
                numInc += 1

        if random() > .5:
            numInc = -numInc

        return NoteChromosome([NoteAllele(toMutate.alleles[0].value + numInc)], self.incProb, self.maxInc)

    def copy(self):
        """ makes a copy of the allele and the chromosome """
        newAlleles = []
        for allele in self.alleles:
            newAlleles.append(allele.copy())
        return NoteChromosome(newAlleles, self.incProb, self.maxInc)

    def __str__(self):
        return str(self.alleles[0])

class GrainChromosome(Chromosome):
    """ grain chromosome, can hold one grain allele """

    def __init__(self, alleles, incrementProbability, maxIncrement):
        Chromosome.__init__(self, alleles)
        
        self.incProb = incrementProbability # chance of point mutation
        self.maxInc = maxIncrement # maximum point mutation

        self.MIN_PITCH = -60
        self.MAX_PITCH = 60
        self.MIN_DURATION = 10
        self.MAX_DURATION = 500
        self.MIN_AMPLITUDE = 0
        self.MAX_AMPLITUDE = 1

    def pitchFitness(self, target):
        """ calculates the pitch fitness by comparing to target """
        return 1 - (abs(self.alleles[0].values[target.alleles[0].PITCH] - target.alleles[0].values[target.alleles[0].PITCH]) / float(self.MAX_PITCH - self.MIN_PITCH))

    def durationFitness(self, target):
        """ calculates the duration fitness by comparing to target """
        return 1 - (abs(self.alleles[0].values[target.alleles[0].DURATION] - target.alleles[0].values[target.alleles[0].DURATION]) / float(self.MAX_DURATION-self.MIN_DURATION))

    def amplitudeFitness(self, target):
        """ calculates the amplitude fitness by comparing to target """
        return 1 - (abs(self.alleles[0].values[target.alleles[0].AMPLITUDE] - target.alleles[0].values[target.alleles[0].AMPLITUDE]) / float(self.MAX_AMPLITUDE-self.MIN_AMPLITUDE))
        
    def fitness(self, target):
        """ calculates the fitness by comparing to target """
        #return self.pitchFitness(target) / 4.0 + self.durationFitness(target) / 3.0 + self.amplitudeFitness(target) / 3.0 # additive fitness function
        return min([self.pitchFitness(target),self.durationFitness(target),self.amplitudeFitness(target)]) # min fitness function

    def cross(self, other):
        """ produces an allele, which is the cross of this chromosome and another chromosome """
        myAllele = self.alleles[0]
        otherAllele = other.alleles[0]
        rand = random()
        if random <= .4:
            return myAllele.copy()
        elif rand <= .6:
            attributes = []
            for i in range(len(myAllele.values)):
                if random() <= .5: attributes.append(myAllele.values[i])
                else: attributes.append(otherAllele.values[i])
            return GrainAllele(attributes[0],attributes[1],attributes[2])
        else:
            return otherAllele.copy()

    def incrementDistance(self):
        """ helper function to determine point mutation """
        numInc = 0
        for i in range(self.maxInc):
            if random() < self.incProb:
                numInc += 1
        return numInc

    def mutate(self, other):
        """ returns a mutated copy of the offspring from this chromosome and another chromosome """
        offAllele = self.cross(other)

        pitchIncrement = self.incrementDistance()
        if random() <= .5:
            pitchIncrement = -pitchIncrement
        durationIncrement = self.incrementDistance() * (self.MAX_DURATION-self.MIN_DURATION) / float(self.MAX_PITCH - self.MIN_PITCH)
        if random() <= .5:
            durationIncrement = -durationIncrement
        amplitudeIncrement = self.incrementDistance() * (self.MAX_AMPLITUDE-self.MIN_AMPLITUDE) / float(self.MAX_PITCH - self.MIN_PITCH)
        if random() <= .5:
            amplitudeIncrement = -amplitudeIncrement

        newPitch = offAllele.values[offAllele.PITCH] + pitchIncrement
        newDuration = offAllele.values[offAllele.DURATION] + durationIncrement
        newAmplitude = offAllele.values[offAllele.AMPLITUDE] + amplitudeIncrement

        if newPitch < self.MIN_PITCH:
            newPitch = self.MIN_PITCH
        if newPitch > self.MAX_PITCH:
            newPitch = self.MAX_PITCH

        if newDuration < self.MIN_DURATION:
            newDuration = self.MIN_DURATION
        if newDuration > self.MAX_DURATION:
            newDuration = self.MAX_DURATION

        if newAmplitude < self.MIN_AMPLITUDE:
            newAmplitude = self.MIN_AMPLITUDE
        if newAmplitude > self.MAX_AMPLITUDE:
            newAmplitude = self.MAX_AMPLITUDE
        
        offAllele = GrainAllele(newPitch, newDuration, newAmplitude)

        newChromosome = GrainChromosome([offAllele], self.incProb, self.maxInc)
            
        newChromosome.checkRep()

        return newChromosome

    def copy(self):
        """ makes a copy of all of the alleles and the chromosome """
        return GrainChromosome([self.alleles[0].copy()],self.incProb,self.maxInc)

    def checkRep(self):
        """ debugging purposes """
        if len(self.alleles) == 1: # should always just store one allele
            # check ranges
            if (self.alleles[0].values[0] >= self.MIN_PITCH and self.alleles[0].values[0] <= self.MAX_PITCH):
                if (self.alleles[0].values[1] >= self.MIN_DURATION and self.alleles[0].values[1] <= self.MAX_DURATION):
                    if (self.alleles[0].values[2] >= self.MIN_AMPLITUDE and self.alleles[0].values[2] <= self.MAX_AMPLITUDE):
                        return
                    raise Exception("grain check rep exception. amplitude range" + str(self.alleles[0].values[2]))
                raise Exception("grain check rep exception. duration range" + str(self.alleles[0].values[1]))
            raise Exception("grain check rep exception. pitch range: " + str(self.alleles[0].values[0]))
        raise Exception("grain check rep exception. length")

    def __str__(self):
        return str(self.alleles[0])

class ChordChromosome(Chromosome):
    """ chord chromosome, can hold one or more note alleles """
    
    def __init__(self, alleles, incrementProbabilityAll, maxIncrementAll, incrementProbability, maxIncrement):
        Chromosome.__init__(self,alleles)
        self.incProbAll = incrementProbabilityAll # chance of shift mutation
        self.maxIncAll = maxIncrementAll # maximum shift mutation
        self.incProb = incrementProbability # chance of point mutation
        self.maxInc = maxIncrement # maximum point mutation
        
        self.MAX_INTERVAL = 8 # maximum interval between two adjacent notes in chord
        self.MIN_INTERVAL = 0 # minimum interval between two adjacent notes in chord

    def allelesToIntervals(self):
        """ finds the interval between each allele """
        return [self.alleles[i+1].value - self.alleles[i].value for i in range(0,len(self.alleles)-1)]

    def fitness(self, target):
        """ calculates the fitness by comparing to target """
        # first, compare intervals between each allele
        myIntervals = self.allelesToIntervals()
        targetIntervals = target.allelesToIntervals()

        numDifIntervals = 0
        for i in range(len(myIntervals)):
            if myIntervals[i] != targetIntervals[i]:
                numDifIntervals += 1

        fracSameIntervals = 1 - float(numDifIntervals) / len(myIntervals)

        # then, compare starting note
        startingFrac = 1 - float(abs(target.alleles[0].value - self.alleles[0].value))/127

        return .5*fracSameIntervals + .5*startingFrac

    def mutate(self, other):
        """ mutates this chromosome """
        
        if random() > .5: toMutate = self
        else: toMutate = other
        
        newAlleles = []

        # first, apply point mutations to alleles
        for j in range(len(toMutate.alleles)):

            allele = toMutate.alleles[j]
            
            numInc = 0
            for i in range(toMutate.maxInc):
                if random() > toMutate.incProb:
                    numInc += 1

            if random() > .5:
                numInc = -numInc

            newValue = allele.value + numInc

            # make sure that the gap between this note and the previous note is not too small
            if j != 0 and (newValue - newAlleles[j-1].value) < self.MIN_INTERVAL:
                #print "prev too small: " + str(newValue) + "," + str(newAlleles[j-1])
                newValue = newAlleles[j-1].value + self.MIN_INTERVAL

            # make sure that the gap between this note and the previous note is not too large
            if j != 0 and (newValue - newAlleles[j-1].value) > self.MAX_INTERVAL:
                #print "prev too big" + str(newValue) + "," + str(newAlleles[j-1])
                newValue = newAlleles[j-1].value + self.MAX_INTERVAL

            # make sure that the gap between this note and the next note is not too small
            if j != (len(toMutate.alleles)-1) and (toMutate.alleles[j+1].value - newValue) < self.MIN_INTERVAL:
                #print "next too small" + str(newValue) + "," + str(toMutate.alleles[j+1].value)
                newValue = toMutate.alleles[j+1].value - self.MIN_INTERVAL

            # make sure that the gap between this note and the next note is not too large
            if j != (len(toMutate.alleles)-1) and (toMutate.alleles[j+1].value - newValue) > self.MAX_INTERVAL:
                #print "next too big" + str(newValue) + "," + str(toMutate.alleles[j+1].value)
                newValue = toMutate.alleles[j+1].value - self.MAX_INTERVAL
                

            newAlleles.append(NoteAllele(newValue))
            
        # then, shift the entire chord
        numIncAll = 0
        for i in range(toMutate.maxIncAll):
            if random() > toMutate.incProbAll:
                numIncAll += 1
        if random() > .5:
            numIncAll = -numIncAll

        for allele in newAlleles:
            allele.value += numIncAll

        newChromosome = ChordChromosome(newAlleles, self.incProbAll, self.maxIncAll, self.incProb, self.maxInc)
            
        newChromosome.checkRep()

        return newChromosome

    
    def copy(self):
        """ makes a copy of all of the alleles and the chromosome """
        newAlleles = []
        for allele in self.alleles:
            newAlleles.append(allele.copy())
        return ChordChromosome(newAlleles, self.incProbAll, self.maxIncAll, self.incProb, self.maxInc)

    def checkRep(self):
        """ debugging purposes """
        Chromosome.checkRep(self)
        for interval in self.allelesToIntervals():
            if interval < self.MIN_INTERVAL or interval > self.MAX_INTERVAL:
                print "chord check rep violated: " 
                print self.allelesToIntervals()
                for allele in self.alleles:
                    print allele
                raise Exception("chord check rep exception.")

    def __str__(self):
        toReturn = "{chordChromosome: "
        for allele in self.alleles:
            toReturn += str(allele) + ", "
        return toReturn[:-2] + "}"

####################################################################
## POPULATION CODE
####################################################################

class Population:

    def __init__(self, chromosomes, target, chromosomeSelecter):
        self.chromosomes = chromosomes
        self.target = target
        self.totalFitness = 0
        self.numChromosomes = len(chromosomes)
        self.updateTotalFitness()
        self.chromosomeSelecter = chromosomeSelecter

    def updateTotalFitness(self):
        """ Calculates the total fitness of all of the chromosomes """
        self.totalFitness = 0
        for chromosome in self.chromosomes:
            self.totalFitness += chromosome.fitness(self.target)

    def percentFitness(self, chromosome):
        """ The percentage of fitness out of the total population that a chromosome contains """
        return chromosome.fitness(self.target) / float(self.totalFitness)

    def match(self):
        """ Randomly pairs up all of the chromosomes for mating, mutates the chromosome list """
        matches = []
        while (len(self.chromosomes) > 0):
            index = int(random() * len(self.chromosomes))
            chromosome1 = self.chromosomes[index]
            self.chromosomes.remove(chromosome1)

            index = int(random() * len(self.chromosomes))
            chromosome2 = self.chromosomes[index]
            self.chromosomes.remove(chromosome2)

            matches.append((chromosome1,chromosome2))
        return matches

    def select(self, pair, child):
        """ Selects which chromosomes will carry on to the next stage """
        if pair[0].fitness(self.target) > pair[1].fitness(self.target): return (pair[0], child)
        else: return (pair[1], child)

    def checkRep(self):
        """ debugging purposes """
        # the number of chromosomes should never grow!
        if len(self.chromosomes) > self.numChromosomes:
            raise Exception("check rep exception")
        # check each chromosome
        for chromosome in self.chromosomes:
            chromosome.checkRep()

    def step(self):
        """ runs one iteration of the GA """
        chromosomePairs = self.match()
        for pair in chromosomePairs:
            newPair = self.select(pair, pair[0].mutate(pair[1]))
            self.chromosomes.append(newPair[0])
            self.chromosomes.append(newPair[1])
        self.updateTotalFitness()
        self.checkRep()
        return self.chromosomeSelecter.select(self.chromosomes,self.target)

    def __str__(self):
        toReturn = "Chromosomes: "
        for chromosome in self.chromosomes:
            toReturn += str(chromosome) + ","
        return toReturn[:-1]


####################################################################
## FUNCTIONS TO PICK CHROMOSOME
####################################################################

class ChromosomeSelecter:
    """ Base class for chromosome selecter """
    
    def __init__(self):
        pass

    def select(self, chromosomes, target):
        raise Exception("not implemented")

class HybridChromosomeSelecter(ChromosomeSelecter):
    """ Combines two chromosome selecters """
    
    def __init__(self, selecters):
        self.selecters = selecters

    def select(self, chromosomes, target):
        toReturn = []
        for selecter in self.selecters:
            toReturn.append(selecter.select(chromosomes, target))
        return toReturn

class WorstChromosomeSelecter(ChromosomeSelecter):
    """ Selects the least fit chromosome """
    
    def __init__(self):
        ChromosomeSelecter.__init__(self)

    def select(self, chromosomes, target):
        bestFitness = 0
        bestChromosome = None
        for chromosome in chromosomes:
            curFitness = chromosome.fitness(target)
            if bestChromosome == None or curFitness <= bestFitness:
                bestFitness = curFitness
                bestChromosome = chromosome
        return bestChromosome
        

class BestChromosomeSelecter(ChromosomeSelecter):
    """ Selects the most fit chromosome """
    
    def __init__(self):
        ChromosomeSelecter.__init__(self)

    def select(self, chromosomes, target):
        bestFitness = 0
        bestChromosome = None
        for chromosome in chromosomes:
            curFitness = chromosome.fitness(target)
            if curFitness >= bestFitness:
                bestFitness = curFitness
                bestChromosome = chromosome
        return bestChromosome

class RandomChromosomeSelecter(ChromosomeSelecter):
    """ Selects a random chromosome """
    
    def __init__(self):
        ChromosomeSelecter.__init__(self)

    def select(self, chromosomes, target):
        return chromosomes[int(random() * len(chromosomes))]
        
class MultipleChromosomeSelecter(ChromosomeSelecter):
    """ Selects multiple chromosomes according to chromosome selecter """

    def __init__(self, chromosomeSelecter, n):
        ChromosomeSelecter.__init__(self)
        self.chromosomeSelecter = chromosomeSelecter
        self.n = n

    def select(self, chromosomes, target):
        chCopy = []
        for chromosome in chromosomes:
            chCopy.append(chromosome.copy())

        toReturn = []
        for i in range(self.n):
            chromosome = self.chromosomeSelecter.select(chCopy, target)
            chCopy.remove(chromosome)
            toReturn.append(chromosome)
        return toReturn

####################################################################
## FUNCTIONS TO DETERMINE WHEN TO STOP
####################################################################

def bestFitness(population, fitnessThreshold):
    """ Stop once the best chromosome is within a fitness threshold """
    return BestChromosomeSelecter().select(population.chromosomes, population.target).fitness(population.target) >= fitnessThreshold

def averageFitness(population, fitnessThreshold):
    """ Stop once the average chromosome is within a fitness threshold """
    return population.totalFitness/float(population.numChromosomes) >= fitnessThreshold

####################################################################
## TRANSITION CODE
####################################################################

def simpleTransition(population, fitnessThreshold):
    """ Steps until the most fit chromosome reaches a fitness threshold """
    
    samples = []
    while not bestFitness(population, fitnessThreshold):
        chrome = population.step()
        samples.append(chrome)

    return samples
    
def transition(source, target, numChromosomes, fitnessThreshold, selecter):
    """ Initializes a population and then calls simple transition """
    
    chromosomes = []
    for i in range(numChromosomes):
        chromosomes.append(source.copy())
    population = Population(chromosomes, target, selecter)

    return simpleTransition(population, fitnessThreshold)

def multipleTransitions(source, targets, numChromosomes, fitnessThresholds):
    """ Initializes a population and then calls simple transition.
        Uses the previous population on another simple transition.
        Repeats until all transitions completed """
    
    chromosomes = []
    for i in range(numChromosomes):
        chromosomes.append(source.copy())
    population = Population(chromosomes, targets[0], RandomChromosomeSelecter())

    path = []

    path.append(simpleTransition(population,fitnessThresholds[0]))
    
    for j in range(1,len(targets)):
        population = Population(population.chromosomes, targets[j], RandomChromosomeSelecter())
        path.append(simpleTransition(population, fitnessThresholds[j]))

    return path

def testSingleTransition():
    """ An example / test of how we can use the note chromosome in a single transition """
    
    source = NoteChromosome([NoteAllele(45)], .5, 3)
    target = NoteChromosome([NoteAllele(65)], .5, 3)

    path = "path: "
    for chromosome in transition(source, target, 10, 1, HybridChromosomeSelecter([WorstChromosomeSelecter(), BestChromosomeSelecter()])):
        path += "[" + str(chromosome[0]) + "," + str(chromosome[1]) + "]" + ","
    print path[:-1] ## get rid of last comma

def testMultipleTransitions():
    """ An example / test of how we can use the note chromosome in multiple transitions """
    
    source = NoteChromosome([NoteAllele(45)], .5, 3)
    target = NoteChromosome([NoteAllele(65)], .5, 3)
    target2 = NoteChromosome([NoteAllele(35)], .5, 3)
    
    for trans in multipleTransitions(source, [target, target2], 10, [.95,.95]):

        path = "path: "
        for chromosome in trans:
            path += str(chromosome) + ","
        print path[:-1] ## get rid of last comma

def testSingleTransitionChord():
    """ An example / test of how we can use the chord chromosome in a single transitions """
        
    source = ChordChromosome([NoteAllele(45), NoteAllele(47), NoteAllele(50)], .5, 3, .5, 3)
    target = ChordChromosome([NoteAllele(55), NoteAllele(60), NoteAllele(61)], .5, 3, .5, 3)
    
    path = "path: "
    for chromosome in transition(source, target, 10, 1, RandomChromosomeSelecter()):
        path += str(chromosome) + ","
    print path[:-1] ## get rid of last comma

def toAttribute(chromosomes, index):
    """ A helper function to extract all of one attribute from granular synthesis chromosomes """
    
    return [chromosome.alleles[0].values[index] for chromosome in chromosomes]

def generateSample1():
    """ Code to generate sample 1 """
    
    source = GrainChromosome([GrainAllele(-15,200,.5)], .05, 3)
    target = GrainChromosome([GrainAllele(40,25,1.0)], .05, 3)

    chromosomes = transition(source, target, 10, .9, RandomChromosomeSelecter())

    path = "path: "
    steps = 0
    for chromosome in chromosomes:
        path += str(chromosome) + ','
        steps += 1
        
    print "number of steps: " + str(steps)
    print "pitches: " + str(toAttribute(chromosomes,0))
    print "durations: " + str(toAttribute(chromosomes,1))

    ampStr = ""
    for amp in toAttribute(chromosomes,2):
        ampStr += str((round(amp,2))) + ","
    print ampStr[:-1]

    playPitchSequencesWithDurations([[str(attri) for attri in toAttribute(chromosomes,0)]], [[dur/1000.0 for dur in toAttribute(chromosomes,1)]], [ampStr[:-1]], [4])
    
def generateSample2():
    """ Code to generate sample 2 """
    source = GrainChromosome([GrainAllele(-15,200,.5)], .5, 3)
    target = GrainChromosome([GrainAllele(40,25,1.0)], .5, 3)

    chromosomes = transition(source, target, 10, .9, BestChromosomeSelecter())

    path = "path: "
    steps = 0
    for chromosome in chromosomes:
        path += str(chromosome) + ','
        steps += 1
    print "pitches: " + str(toAttribute(chromosomes,0))
    print "durations: " + str(toAttribute(chromosomes,1))
    print "number of steps: " + str(steps) + path[:-1] + "\n"

    
    ampStr = ""
    for amp in toAttribute(chromosomes,2):
        ampStr += str((round(amp,2))) + ","
    print ampStr[:-1]

    playPitchSequencesWithDurations([[str(attri) for attri in toAttribute(chromosomes,0)]], [[dur/1000.0 for dur in toAttribute(chromosomes,1)]], [ampStr[:-1]], [4])

def generateSample3():
    """ Code to generate sample 3 """
    source = GrainChromosome([GrainAllele(40,25,.5)], .5, 3)
    target = GrainChromosome([GrainAllele(-10,200,1.0)], .5, 3)

    chromosomePairs = transition(source, target, 10, .9, HybridChromosomeSelecter([WorstChromosomeSelecter(), BestChromosomeSelecter()]))

    best = [chromosomePair[1] for chromosomePair in chromosomePairs]
    worst = [chromosomePair[0] for chromosomePair in chromosomePairs]

    path = "path: "
    for chromosomes in chromosomePairs:
        path += "[" + str(chromosomes[0]) + "," + str(chromosomes[1]) + "]" + ","
    print path[:-1]

    bestAmpStr = ""
    for amp in toAttribute(best,2):
        bestAmpStr += str((round(amp,2))) + ","

    worstAmpStr = ""
    for amp in toAttribute(worst,2):
        worstAmpStr += str((round(amp,2))) + ","

    playPitchSequencesWithDurations([[str(attri) for attri in toAttribute(best,0)],[str(attri) for attri in toAttribute(worst,0)]],
                                    [[dur/1000.0 for dur in toAttribute(best,1)], [dur/1000.0 for dur in toAttribute(worst,1)]],
                                    [bestAmpStr[:-1],worstAmpStr[:-1]],
                                    [4,4])
                                    
                                    
## Uncomment these to generate one of the three samples
generateSample1()
generateSample2()
generateSample3()
