package ui;

import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.util.Set;

import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.event.MouseInputAdapter;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeSelectionModel;

import photo.Photo;
import photo.PhotoLoader;

/**
 * PhotoOrganizer is a window that allows arranging photos into
 * hierarchical albums and viewing the photos in each album.
 * 
 * @author rcm
 */
public class PhotoOrganizer extends JFrame  {

	private final JTree albumTree;
	private final PreviewPane previewPane;
	
	/**
	 * Main entry point of photo organizer.
	 * @param args command line arguments
	 */
	public static void main(final String[] args) {
		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				PhotoOrganizer main = new PhotoOrganizer();
				
				main.setVisible(true);
				
				if (args.length == 0) {
					main.loadPhotos("sample-photos");
				} else if (args.length == 1) {
					main.loadPhotos(args[0]);
				} else {
					System.err.println("too many command-line arguments");
					System.exit(0);
				}
			}
		});
	}
	
	/**
	 * Make a PhotoOrganizer window.
	 */
	public PhotoOrganizer() {
		
		// set up the panel on the left with two subpanels in a vertical layout
		JPanel catalogPanel = new JPanel();
		catalogPanel.setLayout(new BoxLayout(catalogPanel,
				BoxLayout.PAGE_AXIS));
		
		// make the row of buttons 
		JPanel buttonPanel = makeButtonPanel();
		catalogPanel.add(buttonPanel);
		
		// make the album tree
		albumTree = makeCatalogTree();
		catalogPanel.add(new JScrollPane(albumTree));
		
		// make the image previewer
		previewPane = new PreviewPane();

		// put the catalog tree and image previewer side by side, 
		// with an adjustable splitter between
		JSplitPane splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT,
				catalogPanel, previewPane);
		splitPane.setDividerLocation(450);
		this.add(splitPane);
		
		// give the whole window a good default size
		this.setTitle("Photo Organizer");
        this.setSize(900,600);

        // end the program when the user presses the window's Close button
		this.setDefaultCloseOperation(EXIT_ON_CLOSE);		
	}

	/**
	 * Load the photos found in all subfolders of a path on disk.
	 * If path is not an actual folder on disk, has no effect.
	 */
	public void loadPhotos(String path) {
		Set<Photo> photos = PhotoLoader.loadPhotos(path);
		previewPane.display(photos);
	}
	
	/**
	 * Make the button panel for manipulating albums and photos.
	 */
	private JPanel makeButtonPanel() {
		JPanel panel = new JPanel();
		
		// Using a BoxLayout so that buttons will be horizontally aligned
		panel.setLayout(new BoxLayout(panel, BoxLayout.LINE_AXIS));
		
		JButton newAlbumButton = new JButton("New Album");
		newAlbumButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				// REPLACE WITH YOUR CODE
				String newAlbumName = promptForAlbumName();
				if (newAlbumName == null) return;
				System.out.println("new album " + newAlbumName + " as subalbum of " + getSelectedTreeNode());
			}
		});
		panel.add(newAlbumButton);

		JButton deleteAlbumButton = new JButton("Delete Album");
		deleteAlbumButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				// REPLACE WITH YOUR CODE
				System.out.println("delete album " + getSelectedTreeNode());
			}
		});
		panel.add(deleteAlbumButton);

		JButton addPhotosButton = new JButton("Add Photos");
		addPhotosButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				// REPLACE WITH YOUR CODE
				System.out.println("add " + previewPane.getSelectedPhotos().size() 
								   + " photos to album " + getSelectedTreeNode());
			}
		});
		panel.add(addPhotosButton);

		JButton removePhotosButton = new JButton("Remove Photos");
		removePhotosButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				// REPLACE WITH YOUR CODE
				System.out.println("remove " + previewPane.getSelectedPhotos().size() + " photos");
			}
		});
		panel.add(removePhotosButton);

		return panel;
	}

	/**
	 * Make the tree showing album names.
	 */
	private JTree makeCatalogTree() {
		// DELETE OR REPLACE WITH YOUR CODE
		DefaultMutableTreeNode root = new DefaultMutableTreeNode("all photos");
		DefaultMutableTreeNode cambridge;
		root.add(cambridge = new DefaultMutableTreeNode("Cambridge"));
		cambridge.add(new DefaultMutableTreeNode("MIT"));
		cambridge.add(new DefaultMutableTreeNode("Harvard"));
		root.add(new DefaultMutableTreeNode("New York"));
		// end of fake tree
		
		final JTree tree = new JTree(root);
		tree.setMinimumSize(new Dimension(200,400));

		tree.setToggleClickCount(3); // so that we can use double-clicks for previewing instead of expanding/collapsing

		DefaultTreeSelectionModel selectionModel = new DefaultTreeSelectionModel();
		selectionModel.setSelectionMode(DefaultTreeSelectionModel.SINGLE_TREE_SELECTION);
		tree.setSelectionModel(selectionModel);
		
		tree.addMouseListener(new MouseInputAdapter() {
			@Override
			public void mouseClicked(MouseEvent e) {
				// if left-double-click
				if (e.getButton() == MouseEvent.BUTTON1 && e.getClickCount() == 2) {
					// YOUR CODE HERE
					System.out.println("show the photos for album " + getSelectedTreeNode());
				}
			}
		});	
		
		return tree;
	}
	
	/**
	 * Return the album currently selected in the album tree.
	 * Returns null if no selection.
	 */
	private Object getSelectedTreeNode() {
		return albumTree.getLastSelectedPathComponent();
	}

	/**
	 * Pop up a dialog box prompting the user for a name for a new album.
	 * Returns the name, or null if the user pressed Cancel.
	 */
	private String promptForAlbumName() {
		return (String)
		  JOptionPane.showInputDialog(
				albumTree, 
				"Album Name: ", 
				"Add Album",
				JOptionPane.PLAIN_MESSAGE, 
				null, 
				null, 
				"");		
	}
}
