/**
 * author: Slavomir Tuleja, August 18, 2002 - August 31, 2002
 * reworked: Febuary 2003
 * Last change: February 22, 2003
 * coauthors: Tomas Jezo, Jozef Hanc
 * The applet simulates motion of an orbiter moving freely
 * in equatorial plane of a Schwarzschild black hole
 * It is also capable of simulating the case of Kerr black hole ... Just set a!=0 ...
 */

import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import java.applet.*;
import javax.swing.*;
import javax.swing.event.*;


public class GRorbits extends JApplet
   implements Runnable, ActionListener, ChangeListener{

   public BufferedImage image, imageVm;
   public Graphics2D g2, g2Vm;
   private JPanel pnlRight,pnlCenter,pnlLabels,pnlScrollBars,pnlButtons;
   private JPanel pnlDrawingSpeed, pnlStatus, pnlCenterBottom;
   private ImageIcon iconPlay,iconStop,iconStepForward, iconStepBack;
   private MediaTracker tracker;
   private Image imPlay,imStop,imStepForward,imStepBack;
   private JButton btnStartStop, btnStepForward, btnStepBack, btnReset;
   private JLabel lblEffPot, lblT, lblTau;
   public JLabel lblA, lblLm, lblEm, lblR;
   private JLabel lblDrawingSpeed, lblCurrentOrbit, lblCurrentZoom;

   private JSlider slDrawingSpeed;
   public JSlider slA, slLm;

   private JMenuBar mainMenu;
   private JMenu[]  menu = {new JMenu("Orbit"), new JMenu("Display"), new JMenu("Initial"), new JMenu("Zoom"), new JMenu("Speed"), new JMenu("About") };
   private JMenuItem menuItemCircMin, menuItemCircMax;
   private ButtonGroup bgMode;
   public JRadioButtonMenuItem menuItemNewton, menuItemSchwarzschildM, menuItemSchwarzschildL, menuItemRainM, menuItemRainL;

   private JCheckBoxMenuItem menuItemGrid, menuItemTrail, menuItemScale;

   private ButtonGroup bgInitial;
   private JRadioButtonMenuItem menuItemInward, menuItemOutward;

   private ButtonGroup bgZoom;
   private JRadioButtonMenuItem menuItemZoom01, menuItemZoom1, menuItemZoom10, menuItemZoom100;
   private JCheckBoxMenuItem menuItemAutoZoom;

   private ButtonGroup bgSpeed;
   private JRadioButtonMenuItem menuItemFastest, menuItemSlow, menuItemSlowest, menuItemByMouse;

   private JMenuItem menuItemAbout;
   public TextualInputWindow windowA, windowLm, windowEm, windowR;

   public Canvas canvas;
   public CanvasVm canvasVm;
   //...objects used for animation
   public Cursor crDefault, crCross, crHand;
   private Container cp;
   public Scale sc, scVm, scInitial, scVmInitial;
   //...object for transforming from real world coordinates to pixels and vice-versa
   private Thread thread=null;
   //...simulation thread
   private MouseHandlerCanvasVm mouseHandlerCanvasVm;
   private MouseHandlerCanvas mouseHandlerCanvas;
   private MouseHandlerA mouseHandlerA;
   private MouseHandlerLm mouseHandlerLm;
   private MouseHandlerEm mouseHandlerEm;
   private MouseHandlerR mouseHandlerR;

   public GROrbitODE ode;
   //...object encapsulating differential equations
   //governing the motion of an orbiter
   public ODEAdaptiveSolver odeAdaptiveSolver;
   //...ODESolver

   private double t,dt,r,phi,tau;
   private double r0,phi0;
   private double rMax; //maximal r in Effective potential diagram

   public double rInitial;
   public double orbitParameterInitial;
   public String simulationInitial; //containing string describing which simulation was used the last time initial conditions were set

   public boolean stopped=true;//true when simulaton is not running
   public boolean cursorOver=false;
   public int ci,cj; //position of cursor over trajectory plot
   private int delay=5;
   private double gridDivision=2;//division of radial grid in Trajectory plot
   public int numPoints=5000;//number of plotted orbit points
   public OrbitPoint[] orbit= new OrbitPoint[numPoints];


   public void init(){
      //GUI is created here...
      //setting Java look and feel
      try{
         UIManager.setLookAndFeel(UIManager.getCrossPlatformLookAndFeelClassName());
      }
      catch(Exception e){
         System.err.println("Couldn't use the cross-platform "+"look and feel: "+e);
      }

      //content pane, images and canvases...
      cp=getContentPane();
      image=new BufferedImage(300,300,BufferedImage.TYPE_INT_RGB);
      canvas=new Canvas(image,this);
      imageVm=new BufferedImage(200,200,BufferedImage.TYPE_INT_RGB);
      canvasVm=new CanvasVm(imageVm,this);
      //cursors
      crDefault=new Cursor(Cursor.DEFAULT_CURSOR);
      crCross=new Cursor(Cursor.CROSSHAIR_CURSOR);
      crHand=new Cursor(Cursor.HAND_CURSOR);
      canvas.setCursor(crCross);

      //definition of Menu
      mainMenu = new JMenuBar();
      bgMode = new ButtonGroup();
      menuItemSchwarzschildM = new JRadioButtonMenuItem("Schwarzschild m>0",true);
      menuItemSchwarzschildL = new JRadioButtonMenuItem("Schwarzschild m=0",false);
      menuItemRainM = new JRadioButtonMenuItem("Rain m>0",false);
      menuItemRainL = new JRadioButtonMenuItem("Rain m=0",false);
      //menuItemRainL.setEnabled(false);
      menuItemNewton = new JRadioButtonMenuItem("Newton",false);

      //for the following no button group necessary...
      menuItemGrid = new JCheckBoxMenuItem("Show grid",true);
      menuItemTrail = new JCheckBoxMenuItem("Show trail",true);
      menuItemScale = new JCheckBoxMenuItem("Show scale",true);

      bgInitial = new ButtonGroup();
      menuItemInward = new JRadioButtonMenuItem("Inward",true);
      menuItemOutward = new JRadioButtonMenuItem("Outward",false);
      menuItemCircMax = new JMenuItem("Set to unstable circular orbit");
      menuItemCircMin = new JMenuItem("Set to stable circular orbit");

      bgZoom = new ButtonGroup();
      menuItemZoom01 = new JRadioButtonMenuItem("x 0.1",false);
      menuItemZoom1 = new JRadioButtonMenuItem("x 1 (Default)",true);
      menuItemZoom10 = new JRadioButtonMenuItem("x 10",false);
      menuItemZoom100 = new JRadioButtonMenuItem("x 100",false);
      menuItemAutoZoom = new JCheckBoxMenuItem("Auto adjust",false);

      bgSpeed = new ButtonGroup();
      menuItemFastest = new JRadioButtonMenuItem("Fastest (Default)",true);
      menuItemSlow = new JRadioButtonMenuItem("Slow",false);
      menuItemSlowest = new JRadioButtonMenuItem("Slowest",false);
      menuItemByMouse = new JRadioButtonMenuItem("Controlled by Mouse",false);
      menuItemAbout = new JMenuItem("About this program");

      menu[0].add(menuItemSchwarzschildM);
      bgMode.add(menuItemSchwarzschildM);
      menu[0].add(menuItemSchwarzschildL);
      bgMode.add(menuItemSchwarzschildL);
      menu[0].addSeparator();
      menu[0].add(menuItemRainM);
      bgMode.add(menuItemRainM);
      menu[0].add(menuItemRainL);
      bgMode.add(menuItemRainL);
      menu[0].addSeparator();
      menu[0].add(menuItemNewton);
      bgMode.add(menuItemNewton);

      menu[1].add(menuItemGrid);
      menu[1].add(menuItemTrail);
      menu[1].add(menuItemScale);

      bgInitial.add(menuItemInward);
      menu[2].add(menuItemInward);
      bgInitial.add(menuItemOutward);
      menu[2].add(menuItemOutward);
      menu[2].addSeparator();
      menu[2].add(menuItemCircMin);
      menu[2].add(menuItemCircMax);

      menu[3].add(menuItemZoom01);
      bgZoom.add(menuItemZoom01);
      menu[3].add(menuItemZoom1);
      bgZoom.add(menuItemZoom1);
      menu[3].add(menuItemZoom10);
      bgZoom.add(menuItemZoom10);
      menu[3].add(menuItemZoom100);
      bgZoom.add(menuItemZoom100);
      menu[3].addSeparator();
      menu[3].add(menuItemAutoZoom);

      menu[4].add(menuItemFastest);
      bgSpeed.add(menuItemFastest);
      menu[4].add(menuItemSlow);
      bgSpeed.add(menuItemSlow);
      menu[4].add(menuItemSlowest);
      bgSpeed.add(menuItemSlowest);
      menu[4].addSeparator();
      menu[4].add(menuItemByMouse);
      bgSpeed.add(menuItemByMouse);

      menu[5].add(menuItemAbout);

      for(int i = 0; i < menu.length; i++) mainMenu.add(menu[i]);
      setJMenuBar(mainMenu);

      //JLabels...
      lblA = new JLabel("a = ");
      lblLm = new JLabel("L/m = ");
      lblEffPot = new JLabel("E/m and effective potential vs. r for");
      lblT=new JLabel("t = ");
      lblTau=new JLabel("tau = ");
      lblEm=new JLabel("E/m = ");
      lblR=new JLabel("r = ");
      lblDrawingSpeed=new JLabel("Drawing speed: ");
      lblCurrentOrbit=new JLabel("Orbit: Schwarzschild m>0");
      lblCurrentZoom=new JLabel("Zoom: x 1");
      lblDrawingSpeed.setVisible(false);

      //JSliders...
      slA=new JSlider(JSlider.HORIZONTAL,-1000,1000,0);
      //setting INVISIBLE the a-label and a-slider
      slA.setVisible(false);
      lblA.setVisible(false);
      slLm=new JSlider(JSlider.HORIZONTAL,0,20000,3800);
      slDrawingSpeed= new JSlider(JSlider.HORIZONTAL,0,95,95);
      slDrawingSpeed.setVisible(false);

      //Buttons...
      //loading gifs for buttons
      tracker = new MediaTracker(this);
      imPlay = getImage(getCodeBase(), "Play.gif");
      tracker.addImage(imPlay,0);
      imStop = getImage(getCodeBase(), "Stop.gif");
      tracker.addImage(imStop,1);
      imStepForward = getImage(getCodeBase(), "StepForward.gif");
      tracker.addImage(imStepForward,2);
      imStepBack = getImage(getCodeBase(), "StepBack.gif");
      tracker.addImage(imStepBack,3);

      while(!tracker.checkAll(true))
         try
      {
         Thread.sleep(100);
      }
      catch(InterruptedException _ex) {
      }

      iconPlay = new ImageIcon(imPlay);
      iconStop = new ImageIcon(imStop);
      iconStepForward = new ImageIcon(imStepForward);
      iconStepBack = new ImageIcon(imStepBack);

      btnStartStop=new JButton("Start", iconPlay);
      btnStartStop.setToolTipText("Starts, stops, and continues the simulation");
      btnStepForward = new JButton("Step",iconStepForward);
      btnStepForward.setToolTipText("Single step forward");
      btnStepBack = new JButton("Step",iconStepBack);
      btnStepBack.setToolTipText("Single step backward");
      btnReset=new JButton("Reset");
      btnReset.setToolTipText("Resumes initially set initial conditions");

      canvasVm.setToolTipText("Click to set new initial conditions");
      //labels panel
      pnlLabels=new JPanel();
      pnlLabels.setLayout(new GridLayout(2,2));
      pnlLabels.add(lblT);
      pnlLabels.add(lblEm);
      pnlLabels.add(lblTau);
      pnlLabels.add(lblR);
      pnlLabels.setPreferredSize(new Dimension(200,40));
      pnlLabels.setAlignmentX(Component.LEFT_ALIGNMENT);


      lblA.setAlignmentX(Component.LEFT_ALIGNMENT);
      lblLm.setAlignmentX(Component.LEFT_ALIGNMENT);
      lblEffPot.setAlignmentX(Component.LEFT_ALIGNMENT);
      canvasVm.setAlignmentX(Component.LEFT_ALIGNMENT);
      slA.setAlignmentX(Component.LEFT_ALIGNMENT);
      slLm.setAlignmentX(Component.LEFT_ALIGNMENT);
      canvasVm.setPreferredSize(new Dimension(200,200));


      //panel pnlScrollBars
      pnlScrollBars = new JPanel();
      pnlScrollBars.setLayout(new BoxLayout(pnlScrollBars,BoxLayout.Y_AXIS));
      pnlScrollBars.add(Box.createRigidArea(new Dimension(0,5)));
      pnlScrollBars.add(lblEffPot);
      pnlScrollBars.add(lblA);
      pnlScrollBars.add(slA);
      pnlScrollBars.add(Box.createRigidArea(new Dimension(0,5)));
      pnlScrollBars.add(lblLm);
      pnlScrollBars.add(slLm);
      pnlScrollBars.add(Box.createRigidArea(new Dimension(0,10)));
      pnlScrollBars.setPreferredSize(new Dimension(200,103));
      pnlScrollBars.setAlignmentX(Component.LEFT_ALIGNMENT);

      //right panel...
      pnlRight = new JPanel();
      pnlRight.setLayout(new BoxLayout(pnlRight,BoxLayout.Y_AXIS));
      pnlRight.setBorder(BorderFactory.createEmptyBorder(0, 10, 0, 0));
      pnlRight.add(canvasVm);
      pnlRight.add(pnlScrollBars);
      pnlRight.add(pnlLabels);

      //panel pnlDrawingSpeed
      pnlDrawingSpeed=new JPanel();
      pnlDrawingSpeed.setLayout(new BoxLayout(pnlDrawingSpeed,BoxLayout.X_AXIS));
      pnlDrawingSpeed.add(lblDrawingSpeed);
      pnlDrawingSpeed.add(Box.createRigidArea(new Dimension(0,20)));
      pnlDrawingSpeed.add(slDrawingSpeed);
      pnlDrawingSpeed.setAlignmentX(Component.LEFT_ALIGNMENT);
      pnlDrawingSpeed.setPreferredSize(new Dimension(300,20));

      //panel pnlStatus
      pnlStatus=new JPanel();
      pnlStatus.setLayout(new BoxLayout(pnlStatus,BoxLayout.X_AXIS));
      pnlStatus.add(lblCurrentOrbit);
      pnlStatus.add(Box.createHorizontalGlue());
      pnlStatus.add(lblCurrentZoom);
      pnlStatus.setAlignmentX(Component.LEFT_ALIGNMENT);
      pnlStatus.setPreferredSize(new Dimension(300,20));

      //panelCenterBottom
      pnlCenterBottom=new JPanel();
      pnlCenterBottom.setLayout(new BoxLayout(pnlCenterBottom,BoxLayout.Y_AXIS));
      pnlCenterBottom.add(pnlDrawingSpeed);
      pnlCenterBottom.add(pnlStatus);
      pnlCenterBottom.setAlignmentX(Component.LEFT_ALIGNMENT);
      pnlCenterBottom.setPreferredSize(new Dimension(300,40));

      pnlCenter=new JPanel();
      pnlCenter.setLayout(new BoxLayout(pnlCenter,BoxLayout.Y_AXIS));
      canvas.setAlignmentX(Component.LEFT_ALIGNMENT);
      canvas.setPreferredSize(new Dimension(300,300));
      pnlCenter.add(canvas);
      pnlCenter.add(Box.createRigidArea(new Dimension(0,5)));
      pnlCenter.add(pnlCenterBottom);

      pnlButtons=new JPanel();
      pnlButtons.setLayout(new GridLayout(1,4));
      pnlButtons.add(btnStartStop);
      pnlButtons.add(btnStepBack);
      pnlButtons.add(btnStepForward);
      pnlButtons.add(btnReset);

      //everything added to content pane...
      cp.setLayout(new BorderLayout());
      cp.add(pnlCenter,BorderLayout.CENTER);
      cp.add(pnlButtons,BorderLayout.SOUTH);
      cp.add(pnlRight, BorderLayout.EAST);

      //event listeners...
      // action listeners
      btnStartStop.addActionListener(this);
      btnStepForward.addActionListener(new StepForwardHandler(this));
      btnStepBack.addActionListener(new StepBackHandler(this));

      btnReset.addActionListener(this);
      menuItemNewton.addActionListener(this);
      menuItemSchwarzschildM.addActionListener(this);
      menuItemSchwarzschildL.addActionListener(this);
      menuItemRainM.addActionListener(this);
      menuItemRainL.addActionListener(this);
      menuItemGrid.addActionListener(this);
      menuItemTrail.addActionListener(this);
      menuItemScale.addActionListener(this);
      menuItemInward.addActionListener(this);
      menuItemOutward.addActionListener(this);
      menuItemCircMin.addActionListener(this);
      menuItemCircMax.addActionListener(this);
      menuItemZoom01.addActionListener(this);
      menuItemZoom1.addActionListener(this);
      menuItemZoom10.addActionListener(this);
      menuItemZoom100.addActionListener(this);
      menuItemAutoZoom.addActionListener(this);
      menuItemFastest.addActionListener(this);
      menuItemSlow.addActionListener(this);
      menuItemSlowest.addActionListener(this);
      menuItemByMouse.addActionListener(this);
      menuItemAbout.addActionListener(this);
      // adjustment listeners
      slA.addChangeListener(this);
      slLm.addChangeListener(this);
      slDrawingSpeed.addChangeListener(this);
      // mouse listeners
      mouseHandlerCanvasVm=new MouseHandlerCanvasVm(this);
      canvasVm.addMouseListener(mouseHandlerCanvasVm);
      canvasVm.addMouseMotionListener(mouseHandlerCanvasVm);

      mouseHandlerCanvas= new MouseHandlerCanvas(this);
      canvas.addMouseMotionListener(mouseHandlerCanvas);
      canvas.addMouseListener(mouseHandlerCanvas);
      // TEXTUAL INPUT listeners
      mouseHandlerA= new MouseHandlerA(this);
      lblA.addMouseListener(mouseHandlerA);
      mouseHandlerLm= new MouseHandlerLm(this);
      lblLm.addMouseListener(mouseHandlerLm);
      mouseHandlerEm= new MouseHandlerEm(this);
      lblEm.addMouseListener(mouseHandlerEm);
      mouseHandlerR= new MouseHandlerR(this);
      lblR.addMouseListener(mouseHandlerR);
      g2=image.createGraphics();
      g2Vm=imageVm.createGraphics();

      //setting up the GROrbitODE and other variables for simulation
      ode=new SchwarzschildMatter(this);
      rMax=25;
      r=20;
      phi=0;
      ode.setIC(r,phi);
      ode.setEm(ode.getVm(r)+1e-15);
      ode.setLm(3.8);
      setScales(rMax);
      t=0;tau=0;
      gridDivision=2;
      dt=1;
      initializeODESolver();
      //The following statement HAS TO BE here TWICE!!!
      ode.initialSetup(); ode.initialSetup();

      //seting default values for INITIAL settings
      rInitial=r;
      orbitParameterInitial=ode.getOrbitParameter();
      scInitial=sc;
      scVmInitial=scVm;
      simulationInitial="matter";

      //TextualInputWindows...
      windowA= new TextualInputWindow(this, "a", " in units of M");
      windowLm= new TextualInputWindow(this, "L/m", " in units of M");
      windowEm= new TextualInputWindow(this, "E/m", " ");
      windowR= new TextualInputWindow(this, "r", " in units of M");

      repaintAll();
      setVisible(true);
   }

   public void repaintAll(){
   //draws bottom and top layers of both Trajectory and Effective potential plot

      //drawing bottom layers of plots
      drawBottomLayer(g2);
      drawBottomLayerVm(g2Vm);

      //drawing top layers of plots
      canvas.repaint();
      canvasVm.repaint();
      lblA.repaint();
      lblLm.repaint();
      lblR.repaint();
      lblCurrentOrbit.repaint();
      lblCurrentZoom.repaint();
      pnlRight.repaint();
      pnlScrollBars.repaint();
      pnlCenter.repaint();
      pnlLabels.repaint();
      pnlButtons.repaint();
      pnlDrawingSpeed.repaint();
      pnlStatus.repaint();
      pnlCenterBottom.repaint();

   }

   public void initializeODESolver(){
      //inicialization of ODE Solver
      odeAdaptiveSolver=new RK45MultiStep(ode);
      odeAdaptiveSolver.setTolerance(1e-10);
      //...error in one step
      odeAdaptiveSolver.initialize(dt);
      //...size of a fixed step
   }


   public void setScales(double rMax){
      //sets scale object for Trajectory plot that depends only on rMax
      sc=new Scale(-ode.rTorPlot(rMax),ode.rTorPlot(rMax),-ode.rTorPlot(rMax),ode.rTorPlot(rMax),0,300,0,300);
      //sets scale object for Effective potential plot, taking it from ode object
      //since it depends not only on rMax but also on Lm and "a" parameter ...
      scVm=ode.getScale(rMax);
   }

   public double getR(){
      //retruns the actual r-coordinate of the applet
      return r;
   }

   public double getPhi(){
      //returns the actual phi-coordinate of the applet
      return phi;
   }

   public void setR(double r){
      //sets the value of r-coordinete
      this.r=r;
   }

   public void setPhi(double phi){
      //sets the value of phi-coordinate
      this.phi=phi;
   }

   public void setT(double t){
      this.t=t;
   }

   public void setTau(double tau){
      this.tau=tau;
   }

   public int getZoom(){
      //returns an integer depending on selected zoom in the menu ...
      if(menuItemZoom01.isSelected()) return 0;
      else if (menuItemZoom1.isSelected()) return 1;
      else if (menuItemZoom10.isSelected()) return 2;
      else return 3;
   }

   public double getRMin(){
      //returns the minimum possible r-coordinete the orbiter can have
      //above the horizon
      return 0;
   }

   public double getRMax(){
      //returns the maximum value of r-coordinate depending on chosen zoom
     return rMax;
   }

   public double getEmMin(){
      //returns the minimum value of energy the orbiter can have
      return ode.getVm(r);
   }

   public double getEmMax(){
      //returns the maximum value of orbiter energy depending on Effective
      //potential plot...
      return scVm.jToy(0);
   }

   public double getInvBMin(){
      //returns the minimum value of 1/b the photon can have
      return ode.getVm(r);
   }

   public double getInvBMax(){
      //returns the maximum value of 1/b the photon can have, depending
      //on Effective potential plot
      return scVm.jToy(0);
   }

   public void setATextual(double a){
      //this is called after clicking OK in Textual Input Window
      //It sets the value of a-parameter in ode object
      double Lm=ode.getLm();
      slA.setValue(Math.round((float)(a*1000)));
      ode.setLm(Lm);
      ode.setA(a);
      t=0;tau=0;
      phi=0;
      ode.initialSetupTextual();
      //seting default values for INITIAL settings
      rInitial=r;
      orbitParameterInitial=ode.getOrbitParameter();
      scInitial=sc;
      scVmInitial=scVm;
      returnBackToInitial();
   }

   public void setEmTextual(double Em){
      //this is called after clicking OK in Textual Input Window
      //It sets the value of Em-parameter in ode object
      ode.setEm(Em);
      t=0;tau=0;
      phi=0;
      ode.initialSetupTextual();
      //seting default values for INITIAL settings
      rInitial=r;
      orbitParameterInitial=ode.getOrbitParameter();
      scInitial=sc;
      scVmInitial=scVm;
      returnBackToInitial();
   }

   public void setLmTextual(double Lm){
      //this is called after clicking OK in Textual Input Window
      //It sets the value of Lm-parameter in ode object
      double a=ode.getA();
      slLm.setValue(Math.round((float)(Lm*1000)));
      ode.setLm(Lm);
      ode.setA(a);
      t=0;tau=0;
      phi=0;
      ode.initialSetupTextual();
      //seting default values for INITIAL settings
      rInitial=r;
      orbitParameterInitial=ode.getOrbitParameter();
      scInitial=sc;
      scVmInitial=scVm;
      returnBackToInitial();
   }

   public void setRTextual(double r){
      //this is called after clicking OK in Textual Input Window
      //It sets the value of r-parameter in ode object
      this.r=r;
      t=0;tau=0;
      phi=0;
      ode.initialSetupTextual();
      //seting default values for INITIAL settings
      rInitial=r;
      orbitParameterInitial=ode.getOrbitParameter();
      scInitial=sc;
      scVmInitial=scVm;
      returnBackToInitial();
   }

   public void setInvBTextual(double invB){
      //this is called after clicking OK in Textual Input Window
      //It sets the value of 1/b-parameter in ode object
      ode.setInvB(invB);
      t=0;tau=0;
      phi=0;
      ode.initialSetupTextual();
   }

   public void actionPerformed(ActionEvent e){
      //defines the method actionPerformed(...) of interface ActionListener
      if(e.getActionCommand().equals("Stop")){
         //after pressing STOP button
         stopped=true;
	 btnStartStop.setText("Start");
         btnStartStop.setIcon(iconPlay);
         btnStepForward.setEnabled(true);
         btnStepBack.setEnabled(true);
         btnReset.setEnabled(true);
         slA.setEnabled(true);
	 slLm.setEnabled(true);
	 menu[0].setEnabled(true);
	 menu[2].setEnabled(true);
	 menu[3].setEnabled(true);
	 menu[5].setEnabled(true);
      }
      else if(e.getActionCommand().equals("Start")){
         //after pressing START button
         stopped=false;
         btnStartStop.setText("Stop");
         btnStartStop.setIcon(iconStop);
         btnStepForward.setEnabled(false);
         btnStepBack.setEnabled(false);
         btnReset.setEnabled(false);
         slA.setEnabled(false);
         slLm.setEnabled(false);
         menu[0].setEnabled(false);
         menu[2].setEnabled(false);
         menu[3].setEnabled(false);
         menu[5].setEnabled(false);
         thread=new Thread(this);
         thread.start();
      }
      else if(e.getActionCommand().equals("Reset")){
         //after pressing RESET button
         returnBackToInitial();
      }
      else if(e.getActionCommand().equals("Fastest (Default)")){
         //after selecting menu Speed->Fastest
	 delay=5;
	 slDrawingSpeed.setValue(100-delay);
	 slDrawingSpeed.repaint();
	 slDrawingSpeed.setVisible(false);
	 lblDrawingSpeed.setVisible(false);
      }
      else if(e.getActionCommand().equals("Slow")){
         //after selecting menu Speed->Slow
         delay=50;
         slDrawingSpeed.setValue(100-delay);
	 slDrawingSpeed.repaint();
	 slDrawingSpeed.setVisible(false);
	 lblDrawingSpeed.setVisible(false);
      }
      else if(e.getActionCommand().equals("Slowest")){
         //after selecting menu Speed->Slowest
         delay=100;
	 slDrawingSpeed.setValue(100-delay);
	 slDrawingSpeed.repaint();
	 slDrawingSpeed.setVisible(false);
	 lblDrawingSpeed.setVisible(false);
      }
      else if(e.getActionCommand().equals("Controlled by Mouse")){
	 //after selecting menu Speed->ByMouse
	 delay=100-slDrawingSpeed.getValue();
	 slDrawingSpeed.setVisible(true);
	 lblDrawingSpeed.setVisible(true);
	 repaint();
      }
      else if(e.getActionCommand().equals("Inward")){
	 //after selecting menu Initial->Inward
         t=0;tau=0;
  	 ode.setSign(-1);
  	 ode.initialSetup();
      }
      else if(e.getActionCommand().equals("Outward")){
	 //after selecting menu Initial->Outward
         t=0;tau=0;
  	 ode.setSign(1);
  	 ode.initialSetup();
      }
      else if(e.getActionCommand().equals("x 0.1")){
         //after selecting menu Zoom->x 0.1
         lblCurrentZoom.setText("Zoom: x 0.1");
         t=0;tau=0;
         gridDivision=0.25;
         dt=0.1;
         initializeODESolver();
         rMax=2.5;
         r=2.4;
	 phi=0;
  	 ode.initialSetup();
         //seting default values for INITIAL settings
         rInitial=r;
         orbitParameterInitial=ode.getOrbitParameter();
         scInitial=sc;
         scVmInitial=scVm;
         returnBackToInitial();
      }
      else if(e.getActionCommand().equals("x 1 (Default)")){
         //after selecting menu Zoom->x 1 (Default)
         lblCurrentZoom.setText("Zoom: x 1");
         t=0;tau=0;
         gridDivision=2;
	 dt=1;
	 initializeODESolver();
         rMax=25;
         r=20;
	 phi=0;
  	 ode.initialSetup();
         //seting default values for INITIAL settings
         rInitial=r;
         orbitParameterInitial=ode.getOrbitParameter();
         scInitial=sc;
         scVmInitial=scVm;
         returnBackToInitial();
      }
      else if(e.getActionCommand().equals("x 10")){
         //after selecting menu Zoom->About this program
  	 lblCurrentZoom.setText("Zoom: x 10");
         t=0;tau=0;
         gridDivision=16;
         dt=10;
         initializeODESolver();
         rMax=250;
         r=200;
         phi=0;
         ode.initialSetup();
         //seting default values for INITIAL settings
         rInitial=r;
         orbitParameterInitial=ode.getOrbitParameter();
         scInitial=sc;
         scVmInitial=scVm;
         returnBackToInitial();
      }
      else if(e.getActionCommand().equals("x 100")){
         //after selecting menu Zoom->x 100
         lblCurrentZoom.setText("Zoom: x 100");
         t=0;tau=0;
         gridDivision=128;
         dt=200;
         initializeODESolver();
         rMax=2500;
         r=2000;
         phi=0;
         ode.initialSetup();
         //seting default values for INITIAL settings
         rInitial=r;
         orbitParameterInitial=ode.getOrbitParameter();
         scInitial=sc;
         scVmInitial=scVm;
         returnBackToInitial();
      }
      else if(e.getActionCommand().equals("Auto adjust")){
         //do nothing
      }
      else if(e.getActionCommand().equals("Set to stable circular orbit")){
         //after selecting menu popupMenu->Set to local minimum
         initializeODESolver();

         adjustInitialConditionsToCircMin();

         phi=0;
         ode.initialSetup();
         //seting default values for INITIAL settings
         rInitial=r;
         orbitParameterInitial=ode.getOrbitParameter();
         scInitial=sc;
         scVmInitial=scVm;
         returnBackToInitial();
      }
      else if(e.getActionCommand().equals("Set to unstable circular orbit")){
         //after selecting menu popupMenu->Set to local maximum
         initializeODESolver();

         adjustInitialConditionsToCircMax();

         phi=0;
         ode.initialSetup();
         //seting default values for INITIAL settings
         rInitial=r;
         orbitParameterInitial=ode.getOrbitParameter();
         scInitial=sc;
         scVmInitial=scVm;
         returnBackToInitial();
      }
      else if(e.getActionCommand().equals("Newton")){
         //after selecting menu Orbit->Newton
         lblCurrentOrbit.setText("Orbit: Newton");
         slLm.setVisible(true);
         lblLm.setVisible(true);
         lblEffPot.setText("E/m and effective potential vs. r for");
         double oldEm=ode.getOrbitParameter();
         ode=new Newton(this);
         returnBackToInitial();
         if(simulationInitial.equals("matter")) ode.setEm(oldEm-1);
         initializeODESolver();
         ode.initialSetup();
         windowEm= new TextualInputWindow(this, "E/m", " ");
         //seting default values for INITIAL settings
         if(simulationInitial.equals("matter") || simulationInitial.equals("light")){
            rInitial=r;
            orbitParameterInitial=ode.getOrbitParameter();
            scInitial=sc;
            scVmInitial=scVm;
            simulationInitial="newton";
         }

         returnBackToInitial();
      }
      else if(e.getActionCommand().equals("Schwarzschild m>0")){
         //after selecting menu Orbit->Einstein m>0
	 lblCurrentOrbit.setText("Orbit: Schwarzschild m>0");
	 slLm.setVisible(true);
	 lblLm.setVisible(true);
         lblEffPot.setText("E/m and effective potential vs. r for");
	 double oldEm=ode.getOrbitParameter();
         ode=new SchwarzschildMatter(this);
         returnBackToInitial();
         if(simulationInitial.equals("newton")) ode.setEm(oldEm+1);

         initializeODESolver();
         ode.initialSetup();
         windowEm= new TextualInputWindow(this, "E/m", " ");
         //seting default values for INITIAL settings
         if(simulationInitial.equals("light") || simulationInitial.equals("newton")){
            rInitial=r;
            orbitParameterInitial=ode.getOrbitParameter();
            scInitial=sc;
            scVmInitial=scVm;
            simulationInitial="matter";
         }

         returnBackToInitial();
      }
      else if(e.getActionCommand().equals("Schwarzschild m=0")){
         //after selecting menu Orbit->Einstein m=0
         lblCurrentOrbit.setText("Orbit: Schwarzschild m=0");
	 slLm.setVisible(false);
	 lblLm.setVisible(false);
         lblEffPot.setText("M/b and effective potential vs. r");
	 ode=new SchwarzschildLight(this);
         returnBackToInitial();
         ode.setOrbitParameter(200);
         initializeODESolver();
         windowEm= new TextualInputWindow(this, "1/b", " in units 1/M");
         //seting default values for INITIAL settings
         if(simulationInitial.equals("matter") || simulationInitial.equals("newton")){
            rInitial=r;
            orbitParameterInitial=ode.getOrbitParameter();
            scInitial=sc;
            scVmInitial=scVm;
            simulationInitial="light";
         }

         returnBackToInitial();
      }
      else if(e.getActionCommand().equals("Rain m>0")){
         //after selecting menu Orbit->Einstein m>0
         lblCurrentOrbit.setText("Orbit: Rain m>0");
         slLm.setVisible(true);
         lblLm.setVisible(true);
         lblEffPot.setText("E/m and effective potential vs. r for");
         double oldEm=ode.getOrbitParameter();
         ode=new RainMatter(this);
         returnBackToInitial();
         if(simulationInitial.equals("newton")) ode.setEm(oldEm+1);
         initializeODESolver();
         ode.initialSetup();
         windowEm= new TextualInputWindow(this, "E/m", " ");
         //seting default values for INITIAL settings
         if(simulationInitial.equals("light") || simulationInitial.equals("newton")){
            rInitial=r;
            orbitParameterInitial=ode.getOrbitParameter();
            scInitial=sc;
            scVmInitial=scVm;
            simulationInitial="matter";
         }
         returnBackToInitial();
      }
      else if(e.getActionCommand().equals("Rain m=0")){
         //after selecting menu Orbit->Einstein m=0
         lblCurrentOrbit.setText("Orbit: Rain m=0");
         slLm.setVisible(false);
         lblLm.setVisible(false);
         lblEffPot.setText("M/b and effective potential vs. r");
         ode=new RainLight(this);
         returnBackToInitial();
         ode.setOrbitParameter(200);
         initializeODESolver();
         ode.initialSetup();
         windowEm= new TextualInputWindow(this, "1/b", " in units 1/M");
         //seting default values for INITIAL settings
         if(simulationInitial.equals("matter") || simulationInitial.equals("newton")){
            rInitial=r;
            orbitParameterInitial=ode.getOrbitParameter();
            scInitial=sc;
            scVmInitial=scVm;
            simulationInitial="light";
         }
         returnBackToInitial();
      }
      else if(e.getActionCommand().equals("About this program")){
         //after selecting menu About->About this program

         JOptionPane.showMessageDialog(this,"This piece of software is intended to accompany\nthe text written by Edwin F. Taylor and John A. Wheeler,\nEXPLORING BLACK HOLES.\n\nBased on a program by Adam G. Riess and Edwin F. Taylor.\nProfessional advisor: Edmund Bertschinger of MIT.\nJava version:\n(c) Slavom\u00EDr Tuleja, Tom\u00E1\u0161 Je\u017Eo, and Jozef Han\u010D\n\nVersion of February 22, 2003\n\nPlease send any comments to:\ntuleja@stonline.sk\n\nThis applet uses classes from\nOpen Source Physics Education Project\nwww.opensourcephysics.org","About this program",JOptionPane.INFORMATION_MESSAGE );
      }
      else if(e.getActionCommand().equals("Show grid")){
         //after selecting menu Display->Show grid
  	 drawBottomLayer(g2);
	 canvas.repaint();
      }
      else if(e.getActionCommand().equals("Show trail")){
	 //after selecting menu Display->Show trail
  	 drawBottomLayer(g2);
	 canvas.repaint();
      }
      else if(e.getActionCommand().equals("Show scale")){
	 //after selecting menu Display->Show scale
  	 drawBottomLayer(g2);
	 canvas.repaint();
      }

   }

   public void stateChanged(ChangeEvent event){
      //defines the method stateChanged(...) of interface ChangeListener
      JSlider source = (JSlider)event.getSource();

      if(source.equals(slA)) {
         //after adjusting slA
         initializeODESolver();
         ode.setLm(slLm.getValue()/1000.0);
         adjustR();
         ode.initialSetup();
         //seting default values for INITIAL settings
         rInitial=r;
         orbitParameterInitial=ode.getOrbitParameter();
         scInitial=sc;
         scVmInitial=scVm;
         returnBackToInitial();
      }
      else if(source.equals(slLm)){
         //after adjusting slLm
         initializeODESolver();
         ode.setLm(slLm.getValue()/1000.0);
  	 adjustR();
         ode.initialSetup();
         //seting default values for INITIAL settings
         rInitial=r;
         orbitParameterInitial=ode.getOrbitParameter();
         scInitial=sc;
         scVmInitial=scVm;
         returnBackToInitial();
      }
      else if(source.equals(slDrawingSpeed)){
         //after adjusting DrawingSpeed
         delay=100-slDrawingSpeed.getValue();
      }
   }

   public void returnBackToInitial(){
      if(menuItemZoom01.isSelected()) {
         lblCurrentZoom.setText("Zoom: x 0.1");
         dt=0.1;
         rMax=2.5;
         gridDivision=0.25;
      }
      else if(menuItemZoom1.isSelected()) {
         lblCurrentZoom.setText("Zoom: x 1");
         dt=1;
         rMax=25;
         gridDivision=2;
      }
      else if(menuItemZoom10.isSelected()) {
         lblCurrentZoom.setText("Zoom: x 10");
         dt=10;
         rMax=250;
         gridDivision=16;
      }
      else if(menuItemZoom100.isSelected()) {
         lblCurrentZoom.setText("Zoom: x 100");
         dt=200;
         rMax=2500;
         gridDivision=128;
      }
      slLm.setEnabled(true);
      stopped=true;
      t=0;tau=0;

      setR(rInitial);
      setPhi(0);
      setT(0);
      setTau(0);
      scVm=scVmInitial;
      sc=scInitial;
      if(simulationInitial.equals("matter") || simulationInitial.equals("newton")) ode.setEm(orbitParameterInitial);
      else ode.setInvB(orbitParameterInitial);
      //sets orbit parameter Em or 1/b
      ode.initialSetupTextual();
      ode.initialSetupTextual();//TWICE intentionally!

   }

   private void adjustR(){
      //used just in stateChanged(...) method
      //If there's a local minimum in effective potential
      //it sets r-coordinate value to this minimum and sets Em value
      //to its smallest allowed value so that
      //the orbiter follows a circular orbit
      //Otherwise r is set to maximum r-coordinate and Em is set just
      //on the effective potential curve

      //THIS FUNCTIONALITY DEACTIVATED!!!
      /*
      double rr=ode.getRLocalMin(rMax);
      if((rr<ode.getRHorizon())||(rr>rMax)) {
         r=rMax;
      }
      else {
         r=rr;
      }
      */

      t=0;tau=0;
      phi=0;
      ode.initialSetupTextual();
   }

   private void adjustInitialConditionsToCircMin(){
      //If there's a local minimum in effective potential
      //it sets r-coordinate value to this minimum and sets Em value
      //to its smallest allowed value so that
      //the orbiter follows a circular orbit
      //Otherwise nothing is done
      //*****************************************
      //ATENTION: This code is OK ONLY FOR a=0
      //*****************************************
      double rr;
      if(simulationInitial.equals("light")) {
         rr=0;
      }
      else if(simulationInitial.equals("matter")){
         if(1-12/(ode.getLm()*ode.getLm())>0){
            rr=ode.getLm()*ode.getLm()/(2.0)*(1+Math.sqrt(1-12/(ode.getLm()*ode.getLm())));
         }
         else rr=0;
      }
      else {//newton
         rr=ode.getLm()*ode.getLm();
      }

      //rr=ode.getRLocalMin(rMax); //...this was NOT precise enough
      if((rr<ode.getRHorizon())||(rr>rMax)) {
         JOptionPane.showMessageDialog(this,"The effective potential has no local minimum on the diagram!\nOrbiter can not be set to stable circular orbit!","Non existing stable orbit",JOptionPane.WARNING_MESSAGE);
      }
      else {
         r=rr;
         t=0;tau=0;
         phi=0;
         if(simulationInitial.equals("matter") || simulationInitial.equals("newton")) ode.setEm(ode.getVm(r)+1e-15);
         else ode.setInvB(ode.getVm(r)+1e-15);
         ode.initialSetupTextual();

      }

   }

   private void adjustInitialConditionsToCircMax(){
      //If there's a local maximum in effective potential
      //it sets r-coordinate value to this minimum and sets Em value
      //to its smallest allowed value so that
      //the orbiter follows a circular orbit
      //Otherwise nothing is done

      double rr;
      if(simulationInitial.equals("light")) {
         rr=3.0;
      }
      else if(simulationInitial.equals("matter")){
         if(1-12.0/(ode.getLm()*ode.getLm())>=0){
            rr=ode.getLm()*ode.getLm()/(2.0)*(1-Math.sqrt(1-12/(ode.getLm()*ode.getLm())));
         }
         else rr=0;
      }
      else {//newton
         rr=0;
      }

      //rr=ode.getRLocalMax(rMax);
      if((rr<ode.getRHorizon())||(rr>rMax)) {
         JOptionPane.showMessageDialog(this,"The effective potential has no local maximum on the diagram!\nOrbiter can not be set to unstable circular orbit!","Non existing unstable orbit",JOptionPane.WARNING_MESSAGE);
      }
      else {
         r=rr;

         t=0;tau=0;
         phi=0;

         if(simulationInitial.equals("matter") || simulationInitial.equals("newton")) ode.setEm(ode.getVm(r)+1e-15);
         else ode.setInvB(ode.getVm(r)+1e-15);
         ode.initialSetupTextual();

      }
   }

   public boolean isStopped(){
      //returns true when the simulation is stopped
      // false when it is running
      return stopped;
   }

   public void adjustZoom(){
      double p=0.1;
      rMax=rMax-p*(rMax-ode.getR());
      double currZoom=rMax/25.0;
      lblCurrentZoom.setText("Zoom: x ".concat(MyUtils.roundOff(currZoom,5)));
      double exponent=Math.floor(Math.log(rMax)/Math.log(2));
      gridDivision=Math.pow(2,exponent-3);
      if(gridDivision<0.000976562) gridDivision=0.000976562;
      setScales(1.1*rMax);
      double dtSign=odeAdaptiveSolver.getStepSize()/Math.abs(odeAdaptiveSolver.getStepSize());
      double ndt=(rMax/50.0);
      odeAdaptiveSolver.initialize(ndt*dtSign);//sign used to allow simulation backwards

      canvas.repaint();
      canvasVm.repaint();
   }

   public void drawTopLayer(Graphics2D g2){
      //draws the Top Layer of the Orbit Plot

      //draws static limit in light grey
      int rInt=sc.xToi(ode.rTorPlot(2))-sc.xToi(0);
      //...size of static limit in pixels
      g2.setColor(new Color(128,128,128));
      g2.fillOval(150-rInt,150-rInt,2*rInt,2*rInt);

      //draws black hole
      rInt=sc.xToi(ode.rTorPlot(ode.getRHorizon()))-sc.xToi(0);
      //...size of a black hole in pixels
      g2.setColor(Color.black);
      g2.fillOval(150-rInt,150-rInt,2*rInt,2*rInt);

      //Coordinate grid
      if(menuItemGrid.isSelected()) drawGrid(g2);

      //draws INNER Cauchy horizon
      rInt=sc.xToi(ode.rTorPlot(ode.getRInnerHorizon()))-sc.xToi(0);
      //...size of a black hole in pixels
      g2.setColor(Color.gray);
      g2.drawOval(150-rInt,150-rInt,2*rInt,2*rInt);

      //draws ring singularity
      rInt=sc.xToi(ode.rTorPlot(0))-sc.xToi(0);
      //...size of a black hole in pixels
      g2.setColor(new Color(150,0,0));
      g2.fillOval(150-rInt,150-rInt,2*rInt,2*rInt);

      //drawing orbit
      if(menuItemTrail.isSelected()) drawTrajectory(g2);

      //drawing the orbiter
      drawOrbiter(g2);

      //Scale
      if(menuItemScale.isSelected()) drawScale(g2);

      //draws black rectangular line around background
      g2.setColor(Color.black);
      g2.drawRect(0,0,299,299);


      g2.setColor(Color.black);
      lblT.setText("t = ".concat(MyUtils.roundOff(t,2)).concat(" M"));
      lblTau.setText("tau = ".concat(MyUtils.roundOff(tau,2)).concat(" M"));
      lblR.setText("r = ".concat(MyUtils.roundOff(r,6)).concat(" M"));
      if(menuItemSchwarzschildL.isSelected()||menuItemRainL.isSelected())lblEm.setText("1/b = ".concat(MyUtils.roundOff(ode.getOrbitParameter(),6)).concat(" 1/M"));
      else lblEm.setText("E/m = ".concat(MyUtils.roundOff(ode.getEm(),6)));
      if(cursorOver) drawPositionRectangle(ci,cj,g2);
   }

   private void drawOrbiter(Graphics2D g2){
      //draws the orbiter in the Orbit Plot
      g2.setColor(Color.red);
      g2.fillOval(sc.xToi(ode.rTorPlot(r)*Math.cos(phi))-3, sc.yToj(ode.rTorPlot(r)*Math.sin(phi))-3,5,5);
   }

   private void drawGrid(Graphics2D g2){
      //draws the radial grid in the Orbit Plot
      g2.setColor(new Color(200,200,200));
      double rr=0;
      while(rr<1.6*rMax){
         g2.drawOval(sc.xToi(-ode.rTorPlot(rr)),sc.yToj(ode.rTorPlot(rr)),sc.xToi(ode.rTorPlot(rr))-sc.xToi(-ode.rTorPlot(rr)),sc.yToj(-ode.rTorPlot(rr))-sc.yToj(ode.rTorPlot(rr)));
         rr+=4*gridDivision;
      }
      for (int i=0; i<=17; i++){
 	 g2.drawLine(sc.xToi(0),sc.yToj(0),sc.xToi(Math.sqrt(2)*ode.rTorPlot(rMax)*Math.cos(2*Math.PI*i/18f)),sc.yToj(Math.sqrt(2)*ode.rTorPlot(rMax)*Math.sin(2*Math.PI*i/18f)));
      }
   }

   private void drawScale(Graphics2D g2){
      //draws the horizontal linear scale in the Orbit plot
      Font f = new Font("SansSerif", Font.PLAIN,9);
      g2.setFont(f);
      g2.setColor(new Color(200,200,200));
      g2.drawLine(sc.xToi(ode.rTorPlot(ode.getRInnerHorizon())),sc.yToj(0),sc.xToi(ode.rTorPlot(ode.getRHorizon())),sc.yToj(0));
      g2.setColor(Color.black);
      g2.drawLine(sc.xToi(ode.rTorPlot(ode.getRHorizon())),sc.yToj(0),sc.xToi(ode.rTorPlot(1.1*rMax)),sc.yToj(0));
      double rr=0;
      while(rr<1.1*rMax){
         if(rr<ode.getRHorizon()) g2.setColor(new Color(200,200,200));
         else g2.setColor(Color.black);
         g2.drawLine(sc.xToi(ode.rTorPlot(rr)),sc.yToj(0)-3,sc.xToi(ode.rTorPlot(rr)),sc.yToj(0)+3);
         rr+=4*gridDivision;
      }
      rr=0;
      double scaleDivision=0.5*Math.pow(2,Math.round((float)(Math.log(0.5*(rMax))/Math.log(2))));
      int digits=-Math.round((float)(Math.log(0.02*scaleDivision)/Math.log(10)));
      if(digits<0) digits=0;
      while(rr<1.1*rMax){
         String s=MyUtils.roundOff(rr,digits);
         FontMetrics fm=g2.getFontMetrics();
         int sW=fm.stringWidth(s)+4;
         int sH=fm.getHeight();
         g2.setColor(new Color(247,247,247));
         g2.fillRect(sc.xToi(ode.rTorPlot(rr))-sW/2,sc.yToj(0)+3,sW,sH);
         //if(rr<ode.getRHorizon()) g2.setColor(new Color(200,200,200));
         //else
         g2.setColor(Color.black);
         g2.drawRect(sc.xToi(ode.rTorPlot(rr))-sW/2,sc.yToj(0)+3,sW,sH);
         MyUtils.drawString(s,sc.xToi(ode.rTorPlot(rr)),sc.yToj(0)+7,2,f,g2);

         rr+=4*gridDivision;
      }
   }

   public void drawBottomLayer(Graphics2D g2){
      //draws background
      g2.setColor(Color.white);
      g2.fillRect(0,0,300,300);

      //draws black rectangular line around background
      g2.setColor(Color.black);
      g2.drawRect(0,0,299,299);
   }

   private void drawTrajectory(Graphics2D g2){

      for(int i=0;i<numPoints-1;i++){
         if(orbit[i].getR()<ode.getRHorizon()) g2.setColor(new Color(255,255,127));
         else g2.setColor(new Color(0,0,127));
         g2.drawLine(sc.xToi(ode.rTorPlot(orbit[i].getR())*Math.cos(orbit[i].getPhi())),sc.yToj(ode.rTorPlot(orbit[i].getR())*Math.sin(orbit[i].getPhi())),sc.xToi(ode.rTorPlot(orbit[i+1].getR())*Math.cos(orbit[i+1].getPhi())),sc.yToj(ode.rTorPlot(orbit[i+1].getR())*Math.sin(orbit[i+1].getPhi())));
      }
   }

   public void drawPositionRectangle(int i,int j, Graphics2D g2){
      //draws blue rectangle in the lower right of trajectory plot
      //with cursor position
      Font f = new Font("SansSerif", Font.PLAIN,12);
      g2.setFont(f);
      double cX=sc.iTox(i);
      double cY=sc.jToy(j);
      double cR=ode.rPlotTor(Math.sqrt(cX*cX+cY*cY));
      if(cR>0){
         double cPhiDeg=(Math.atan2(cY,cX))*180.0/Math.PI;
         if(cPhiDeg<0)cPhiDeg+=360.0;

         g2.setColor(new Color(207,207,255,239));
         g2.fillRect(0,280,159,19);
         g2.setColor(Color.black);
         g2.drawRect(0,280,159,19);
         g2.drawString("r = ".concat(MyUtils.roundOff(cR,2)).concat(" M"),10,295);
         g2.drawString("phi = ".concat(MyUtils.roundOff(cPhiDeg,1)).concat(""),90,295);
      }
      else {
         g2.setColor(new Color(207,207,255));
         g2.fillRect(0,280,159,19);
         g2.setColor(Color.black);
         g2.drawRect(0,280,159,19);
         g2.drawString("Ring Singularity!",10,295);
      }
      canvas.repaint();
   }

   private void drawBottomLayerVm(Graphics2D g2Vm){
      //draws background
      g2Vm.setColor(Color.white);
      g2Vm.fillRect(0,0,200,200);

      //draws black rectangular line around background
      g2Vm.setColor(Color.black);
      g2Vm.drawRect(0,0,199,199);
   }

   public void drawTopLayerVm(Graphics2D g2Vm){
      //drawing Effective potential
      drawEffectivePotential(g2Vm);

      //draws a dot representin orbiter in Eff. potential diagram
      if(ode.getR()<ode.getRHorizon()) g2Vm.setColor(Color.white);
      else g2Vm.setColor(Color.black);
      g2Vm.fillOval(scVm.xToi(r)-3,scVm.yToj(ode.getOrbitParameter())-2,5,5);

      //drawing scale...
      Font f = new Font("SansSerif", Font.PLAIN,9);
      g2Vm.setFont(f);
      if(ode.getR()<ode.getRHorizon())g2Vm.setColor(Color.white);
      else g2Vm.setColor(Color.black);
      double y1=scVm.jToy(200);
      double y2=scVm.jToy(0);
      double scaleDivision=0.5*Math.pow(2,Math.round((float)(Math.log(0.5*(y2-y1))/Math.log(2))));
      int digits=-Math.round((float)(Math.log(0.02*scaleDivision)/Math.log(10)));
      double y=scaleDivision*Math.floor(y1/scaleDivision);

      while(y<y2){
         y+=scaleDivision;
         g2Vm.drawLine(195,scVm.yToj(y),200,scVm.yToj(y));
         FontMetrics fm=g2Vm.getFontMetrics();
         String s=MyUtils.roundOff(y,digits);
         int sW=fm.stringWidth(s)+4;
         int sH=fm.getHeight();
         if((Math.abs(scVm.yToj(y)-scVm.yToj(ode.getOrbitParameter())+2)>=sH/2+1)||( Math.abs(scVm.xToi(ode.getR())-200+2)>=sW+3  )){
            g2Vm.setColor(new Color(247,247,247));
            g2Vm.fillRect(194-sW,scVm.yToj(y)-4,sW,sH);
            g2Vm.setColor(Color.black);
            g2Vm.drawRect(194-sW,scVm.yToj(y)-4,sW,sH);
            MyUtils.drawString(s,193,scVm.yToj(y),1,f,g2Vm);
         }
      }

      //draws black rectangular line around background
      g2Vm.setColor(Color.black);
      g2Vm.drawRect(0,0,199,199);
   }

   public void drawEffectivePotential(Graphics2D g2Vm){
      //drawing shaded area representing ergosphere between horizon and static limit
      g2Vm.setColor(new Color(128,128,128));
      g2Vm.fillRect(1,1,scVm.xToi(2)-1,198);

      //drawing shaded area representing black hole region...
      g2Vm.setColor(Color.black);
      g2Vm.fillRect(1,1,scVm.xToi(ode.getRHorizon())-1,198);

      //draws INNER Cauchy horizon
      int rInt=scVm.xToi(ode.getRInnerHorizon())-scVm.xToi(0);
      //...coordinate of horizon in pixels
      g2Vm.setColor(Color.gray);
      g2Vm.drawLine(rInt,1,rInt,198);

      //drawing the graph...
      double r,dr;
      int jVr,jVrpdr;
      double Vr,Vrpdr;

      dr=rMax/200.0;
      // from inner horizon to singularity
      g2Vm.setColor(Color.white);
      r=ode.getRInnerHorizon()-1e-10;
      Vr=ode.getVm(r);
      Vrpdr=ode.getVm(r-dr);
      while(r>dr){
         if(Vr>10) Vr=100;
         if(Vr<-10) Vr=-10;
         if(Vrpdr>10) Vrpdr=100;
	 if(Vrpdr<-10) Vrpdr=-10;
	 jVr=scVm.yToj(Vr);
	 jVrpdr=scVm.yToj(Vrpdr);

	 g2Vm.drawLine(scVm.xToi(r),jVr,scVm.xToi(r-dr),jVrpdr);
	 r-=dr;
  	 Vr=ode.getVm(r);
	 Vrpdr=ode.getVm(r-dr);
      }
      // from outer horizon to rMax
      g2Vm.setColor(Color.black);
      r=ode.getRHorizon();
      Vr=ode.getVmAtHorizon();
      Vrpdr=ode.getVm(r+dr);
      while(r<1.1*rMax){
         if(Vr>100) Vr=100;
         if(Vr<-5) Vr=-5;
         if(Vrpdr>100) Vrpdr=100;
         if(Vrpdr<-5) Vrpdr=-5;
         jVr=scVm.yToj(Vr);
         jVrpdr=scVm.yToj(Vrpdr);

         g2Vm.drawLine(scVm.xToi(r),jVr,scVm.xToi(r+dr),jVrpdr);
         r+=dr;
           Vr=ode.getVm(r);
         Vrpdr=ode.getVm(r+dr);
      }

      //drawing Energy (or invB) red line
      g2Vm.setColor(Color.red);
      g2Vm.drawLine(0,scVm.yToj(ode.getOrbitParameter()),200,scVm.yToj(ode.getOrbitParameter()));

   }

   public void run(){
      //defines method run() of interface Runnable
      //associated with the Thread thread
      while (!stopped){
         makeOneSingleStep();
         try {
            thread.sleep(delay);
         }
         catch(InterruptedException exception){
            System.out.println("Interrupted...");
         }

      }
   }

   public void makeOneSingleStep(){
      //makes a single step in calculation
      r0=ode.getR();
      phi0=ode.getPhi();
      try{
         t+=odeAdaptiveSolver.step();
      }
      catch(InsufficientConvergenceException e){
         JOptionPane.showMessageDialog(this,"Numerical method failed to converge...","Loss of precision warning",JOptionPane.WARNING_MESSAGE);
         stopped=true;
         cp.repaint();
         btnStartStop.setText("Start");
         btnStartStop.setIcon(iconPlay);
         slA.setEnabled(true);
         slLm.setEnabled(true);
         btnStepForward.setEnabled(true);
         btnStepBack.setEnabled(true);
         btnReset.setEnabled(true);
         menu[0].setEnabled(true);
         menu[2].setEnabled(true);
         menu[3].setEnabled(true);
         menu[5].setEnabled(true);
      }
      //...executes one step

      r=ode.getR();
      phi=ode.getPhi();
      tau=ode.getTau();

      //to stop the simulation when the orbiter falls to singularity
      if(r<=2e-5){
         JOptionPane.showMessageDialog(this,"Orbiter has reached singularity!","Singularity warning",JOptionPane.WARNING_MESSAGE);
         stopped=true;
         cp.repaint();
         btnStartStop.setText("Start");
         btnStartStop.setIcon(iconPlay);
         slA.setEnabled(true);
         slLm.setEnabled(true);
         btnStepForward.setEnabled(true);
         btnStepBack.setEnabled(true);
         btnReset.setEnabled(true);
         menu[0].setEnabled(true);
         menu[2].setEnabled(true);
         menu[3].setEnabled(true);
         menu[5].setEnabled(true);
         r=0;
      }


      for(int i=0; i<numPoints-1; i++){
         orbit[i]=orbit[i+1];
      }

      orbit[numPoints-1]=new OrbitPoint(r,phi);

      if(menuItemAutoZoom.isSelected()) adjustZoom();

      canvas.repaint();
      canvasVm.repaint();


   }


}

class StepForwardHandler implements ActionListener {
   GRorbits applet;

   public StepForwardHandler(GRorbits applet){
      this.applet=applet;
   }

   public void actionPerformed(ActionEvent e){
      applet.makeOneSingleStep();
   }
}

class StepBackHandler implements ActionListener {
   GRorbits applet;

   public StepBackHandler(GRorbits applet){
      this.applet=applet;
   }

   public void actionPerformed(ActionEvent e){
      applet.odeAdaptiveSolver.setStepSize(-applet.odeAdaptiveSolver.getStepSize());
      applet.makeOneSingleStep();
      applet.odeAdaptiveSolver.setStepSize(-applet.odeAdaptiveSolver.getStepSize());
   }
}

class Canvas extends JPanel{
   //used for double-buffering the image
   BufferedImage image;
   GRorbits applet;

   //constructor
   public Canvas(BufferedImage image, GRorbits applet){
      this.image=image;
      this.applet=applet;
   }

   public void paintComponent(Graphics g){
      Graphics2D g2=(Graphics2D)g;
      g2.drawImage(image,null,0,0);
      applet.drawTopLayer(g2);
   }
}


class CanvasVm extends JPanel{
   //used for double-buffering the imageVm
   BufferedImage imageVm;
   GRorbits applet;

   //constructor
   public CanvasVm(BufferedImage imageVm, GRorbits applet){
      this.imageVm=imageVm;
      this.applet=applet;
   }

   public void paintComponent(Graphics g){
      Graphics2D g2=(Graphics2D)g;
      g2.drawImage(imageVm,null,0,0);
      applet.drawTopLayerVm(g2);
   }
}

class MouseHandlerCanvasVm implements  MouseListener, MouseMotionListener {
   //handles mouse actions above CanvasVm
   GRorbits applet;

   //constructor
   public MouseHandlerCanvasVm(GRorbits applet){
      this.applet=applet;
   }

   public void mousePressed(MouseEvent e){
      if(applet.isStopped()){
         int i=e.getX();
         int j=e.getY();
         double r=applet.scVm.iTox(i);
         applet.setR(r);
         applet.setPhi(0);
         applet.setT(0);
         applet.setTau(0);
         applet.ode.setOrbitParameter(j);
         //sets orbit parameter associated with j
         //i.e. Em or 1/b
         applet.ode.initialSetupTextual();

         //remembering initial setting
         applet.rInitial=r;
         applet.orbitParameterInitial=applet.scVm.jToy(j);
         applet.scInitial=applet.sc;
         applet.scVmInitial=applet.scVm;
         if((applet.menuItemSchwarzschildL.isSelected())||(applet.menuItemRainL.isSelected())){
            applet.simulationInitial="light";
         }
         else if((applet.menuItemSchwarzschildM.isSelected())||(applet.menuItemRainM.isSelected())){
            applet.simulationInitial="matter";
         }
         else applet.simulationInitial="newton";
         applet.returnBackToInitial();
      }
   }

   public void mouseDragged(MouseEvent e){
      if(applet.isStopped() ){
         int i=e.getX();
         int j=e.getY();
         double r=applet.scVm.iTox(i);
         applet.setR(r);
         applet.setPhi(0);
         applet.ode.setOrbitParameter(j);
         applet.ode.initialSetupTextual();

         //remembering initial setting
         applet.rInitial=r;
         applet.orbitParameterInitial=applet.scVm.jToy(j);
         applet.scInitial=applet.sc;
         applet.scVmInitial=applet.scVm;
         if((applet.menuItemSchwarzschildL.isSelected())||(applet.menuItemRainL.isSelected())){
            applet.simulationInitial="light";
         }
         else if((applet.menuItemSchwarzschildM.isSelected())||(applet.menuItemRainM.isSelected())){
            applet.simulationInitial="matter";
         }
         else applet.simulationInitial="newton";
         applet.returnBackToInitial();
      }
   }

   public void mouseReleased(MouseEvent e){
      if(applet.isStopped()){
         int i=e.getX();
         int j=e.getY();

         double r=applet.scVm.iTox(i);
         applet.setR(r);
         applet.setPhi(0);
         applet.ode.setOrbitParameter(j);
         applet.ode.initialSetupTextual();

         //remembering initial setting
         applet.rInitial=r;
         applet.orbitParameterInitial=applet.scVm.jToy(j);
         applet.scInitial=applet.sc;
         applet.scVmInitial=applet.scVm;
         if((applet.menuItemSchwarzschildL.isSelected())||(applet.menuItemRainL.isSelected())){
            applet.simulationInitial="light";
         }
         else if((applet.menuItemSchwarzschildM.isSelected())||(applet.menuItemRainM.isSelected())){
            applet.simulationInitial="matter";
         }
         else applet.simulationInitial="newton";
         applet.returnBackToInitial();
      }
   }

   public void mouseEntered(MouseEvent e){
   }

   public void mouseExited(MouseEvent e){
   }

   public void mouseClicked(MouseEvent e){
   }

   public void mouseMoved(MouseEvent e){
   }
 }

class MouseHandlerCanvas implements  MouseListener, MouseMotionListener {
   //handles mouse actions above Canvas
   GRorbits applet;

   //constructor
   public MouseHandlerCanvas(GRorbits applet){
      this.applet=applet;
   }

   public void mousePressed(MouseEvent e){
   }

   public void mouseDragged(MouseEvent e){
   }

   public void mouseReleased(MouseEvent e){
   }

   public void mouseEntered(MouseEvent e){
      applet.cursorOver=true;
      applet.canvas.repaint();
   }

   public void mouseExited(MouseEvent e){
      applet.cursorOver=false;
      applet.canvas.repaint();
   }

   public void mouseClicked(MouseEvent e){
   }

   public void mouseMoved(MouseEvent e){
      applet.ci=e.getX();
      applet.cj=e.getY();
   }
}

class MouseHandlerA implements MouseListener {
   //handles clicking at lblA JLabel
   //which should cause the windowA to open
   //making possible the NUMERICAL input of a...
   GRorbits applet;

   //constructor
   public MouseHandlerA(GRorbits applet){
      this.applet=applet;
   }

   public void mouseClicked(MouseEvent e){
      if(applet.isStopped()){
         applet.windowA.windowUpdate();
         applet.windowA.setLocation();
         applet.windowA.show();
      }
   }

   public void mouseEntered(MouseEvent e){
      //changes the color and cursor of the labelA
      if(applet.isStopped()){
         applet.lblA.setForeground(new Color(0,0,127));
         applet.lblA.setCursor(applet.crHand);
      }
   }

   public void mouseExited(MouseEvent e){
      //changes the color and cursor of the labelA
      if(applet.isStopped()){
         applet.lblA.setForeground(Color.black);
         applet.lblA.setCursor(applet.crDefault);
      }
   }

   public void mousePressed(MouseEvent e){}
   public void mouseReleased(MouseEvent e){}
}

class MouseHandlerLm implements MouseListener {
   //handles clicking at lblLm JLabel
   //which should cause the windowA to open
   //making possible the NUMERICAL input of Lm...
   GRorbits applet;

   //constructor
   public MouseHandlerLm(GRorbits applet){
      this.applet=applet;
   }

   public void mouseClicked(MouseEvent e){
      if(applet.isStopped()){
         applet.windowLm.windowUpdate();
         applet.windowLm.setLocation();
         applet.windowLm.show();
      }
   }

   public void mouseEntered(MouseEvent e){
      //changes the color and cursor of the labelLm
      if(applet.isStopped()){
         applet.lblLm.setForeground(new Color(0,0,127));
         applet.lblLm.setCursor(applet.crHand);
      }
   }

   public void mouseExited(MouseEvent e){
      //changes the color and cursor of the labelLm
      if(applet.isStopped()){
         applet.lblLm.setForeground(Color.black);
         applet.lblLm.setCursor(applet.crDefault);
      }
   }

   public void mousePressed(MouseEvent e){}
   public void mouseReleased(MouseEvent e){}
}

class MouseHandlerEm implements MouseListener {
   //handles clicking at lblEm JLabel
   //which should cause the windowA to open
   //making possible the NUMERICAL input of Em...
   GRorbits applet;

   //constructor
   public MouseHandlerEm(GRorbits applet){
      this.applet=applet;
   }

   public void mouseClicked(MouseEvent e){
      if(applet.isStopped()){
         applet.windowEm.windowUpdate();
         applet.windowEm.setLocation();
         applet.windowEm.show();
      }
   }

   public void mouseEntered(MouseEvent e){
      //changes the color and cursor of the labelEm
      if(applet.isStopped()){
         applet.lblEm.setForeground(new Color(0,0,127));
         applet.lblEm.setCursor(applet.crHand);
      }
   }

   public void mouseExited(MouseEvent e){
      //changes the color and cursor of the labelEm
      if(applet.isStopped()){
         applet.lblEm.setForeground(Color.black);
         applet.lblEm.setCursor(applet.crDefault);
      }
   }

   public void mousePressed(MouseEvent e){}
   public void mouseReleased(MouseEvent e){}
}

class MouseHandlerR implements MouseListener {
   //handles clicking at lblR JLabel
   //which should cause the windowA to open
   //making possible the NUMERICAL input of R...
   GRorbits applet;

   //constructor
   public MouseHandlerR(GRorbits applet){
      this.applet=applet;
   }

   public void mouseClicked(MouseEvent e){
      if(applet.isStopped()){
         applet.windowR.windowUpdate();
         applet.windowR.setLocation();
         applet.windowR.show();
      }
   }

   public void mouseEntered(MouseEvent e){
      //changes the color and cursor of the labelR
      if(applet.isStopped()){
         applet.lblR.setForeground(new Color(0,0,127));
         applet.lblR.setCursor(applet.crHand);
      }
   }

   public void mouseExited(MouseEvent e){
      //changes the color and cursor of the labelR
      if(applet.isStopped()){
         applet.lblR.setForeground(Color.black);
         applet.lblR.setCursor(applet.crDefault);
      }
   }

   public void mousePressed(MouseEvent e){}
   public void mouseReleased(MouseEvent e){}
}

class TextualInputWindow extends JDialog implements ActionListener {
   //This window pops up when the user clicks at one of the
   //following labels: lblA, lblLm, lblR, lblEm
   GRorbits applet;
   JLabel lblLine1, lblLine2, lblValue;
   JTextField tfValue;
   JPanel pnlCenter, pnlValue, pnlButtons;
   JButton btnOK, btnCancel;
   String sWhat, sUnits, sMinValue, sMaxValue, sValue;
   double minValue, maxValue, value;
   //minValue, maxValue are minimum and maximum values allowed for input
   //value is a current value jus input by user...

   //constructor
   public TextualInputWindow(GRorbits applet, String sWhat, String sUnits){
      //By selecting sWhat and sUnits one can make the Textual Input Window
      //show different text on its labels and react differently, according to
      //the value of sWhat string...
      setModal(true);
      setTitle("Textual Input Window");
      //Window will have a vertical size depenting on wheather this is run as an applet or an application
      setSize(250,125);
      setResizable(false);

      this.sWhat=sWhat;
      this.sUnits=sUnits;
      this.applet=applet;

      Container cp=getContentPane();
      btnOK=new JButton("OK");
      btnCancel=new JButton("Cancel");
      lblLine1= new JLabel();
      lblLine2= new JLabel();
      lblValue= new JLabel();
      tfValue= new JTextField();

      windowUpdate();
      pnlValue=new JPanel();
      pnlValue.setLayout(new BoxLayout(pnlValue,BoxLayout.X_AXIS));
      pnlValue.add(lblValue);
      pnlValue.add(tfValue);
      pnlValue.setAlignmentX(Component.LEFT_ALIGNMENT);
      pnlValue.setPreferredSize(new Dimension(250,20));

      pnlCenter=new JPanel();
      pnlCenter.setLayout(new BoxLayout(pnlCenter,BoxLayout.Y_AXIS));
      pnlCenter.add(lblLine1);
      pnlCenter.add(lblLine2);
      pnlCenter.add(pnlValue);
      pnlCenter.add(Box.createVerticalGlue());
      pnlCenter.setBorder(BorderFactory.createEmptyBorder(0, 5, 0, 5));

      pnlButtons = new JPanel();
      pnlButtons.setLayout(new GridLayout(1,2));
      pnlButtons.add(btnOK);
      pnlButtons.add(btnCancel);
      pnlButtons.setBorder(BorderFactory.createEmptyBorder(0, 5, 0, 5));

      cp.add(pnlButtons,BorderLayout.SOUTH);
      cp.add(pnlCenter,BorderLayout.CENTER);

      btnOK.addActionListener(this);
      btnCancel.addActionListener(this);

   }

   public void windowUpdate(){
      //This is used to change the content of labels in
      //textual input window in accord with current situation...
      if(sWhat.equals("a")) {
         minValue=-1; maxValue=1;
      }
      else if(sWhat.equals("L/m")){
         minValue=0; maxValue=20;
      }
      else if(sWhat.equals("r")){
         minValue=applet.getRMin();
         maxValue=applet.getRMax();
      }
      else if(sWhat.equals("E/m")){
         minValue=applet.getEmMin();
         maxValue=applet.getEmMax();
      }
      else if(sWhat.equals("1/b")){
         minValue=applet.getInvBMin();
         maxValue=applet.getInvBMax();
      }
      //minValue and maxValue are going to be rounded off to 6 decimal places
      sMinValue=MyUtils.roundOff(minValue,6);
      sMaxValue=MyUtils.roundOff(maxValue,6);
      minValue=Double.valueOf(sMinValue).doubleValue();
      maxValue=Double.valueOf(sMaxValue).doubleValue();

      lblLine1.setText("Type in a value of ".concat(sWhat).concat(sUnits));
      lblLine2.setText(sMinValue.concat(" <= ").concat(sWhat).concat(" <= ").concat(sMaxValue));
      lblValue.setText(sWhat.concat(" = "));
      tfValue.setText("");
   }

   public void setLocation(){
      //Sets the location of Textual Input Window so that it is
      //centered at the applet...
      Point cpP=applet.getContentPane().getLocationOnScreen();
      int ic=Math.round((float)cpP.getX());
      int jc=Math.round((float)cpP.getY());
      setLocation(new Point(ic+130,jc+130));
   }

   public void actionPerformed(ActionEvent e){
      if(e.getActionCommand().equals("OK")){
         //after pressing OK button...
         sValue=tfValue.getText();
         try {
            value=Double.valueOf(sValue).doubleValue();

            //adjusting value if it is NOT in the allowed interval
            if((minValue<=value)&&(value<=maxValue)){
               if(sWhat.equals("a")) {
                  applet.setATextual(value);
               }
               else if(sWhat.equals("L/m")){
                  applet.setLmTextual(value);
               }
               else if(sWhat.equals("r")){
                  applet.setRTextual(value);
               }
               else if(sWhat.equals("E/m")){
                  applet.setEmTextual(value);
               }
               else if(sWhat.equals("1/b")){
                  applet.setInvBTextual(value);
               }
               hide();
               applet.repaintAll();
            }
            else {
               tfValue.setText("");
               JOptionPane.showMessageDialog(this,"Input value not in allowed interval!\nPlease try again.","Input error",JOptionPane.ERROR_MESSAGE);
               //System.out.println("Value not in required interval...");
            }
         }
         catch(NumberFormatException ex){
            JOptionPane.showMessageDialog(this,"Wrong input format!\nPlease try again.","Input error",JOptionPane.ERROR_MESSAGE);
            //System.out.println("Wrong number format ...");
            tfValue.setText("");
         }
      }
      else {
         //after pressing Cancel button...
         hide();
         applet.repaintAll();
      }
      //set focus back to textfield
      tfValue.requestFocus();

   }

}


class OrbitPoint {
   double r,phi;

   //constructor
   public OrbitPoint(double r, double phi){
      this.r=r;
      this.phi=phi;
   }

   public double getR(){
      return r;
   }

   public double getPhi(){
      return phi;
   }
}
