/** Object encapsulating differential equations for motion of LIGHT
 *  in equatorial plane of a Schwarzschild black hole in RAIN COORDINATES
 *  For comments about the functionality of the methods below,
 *  please see the definition of GROrbitODE interface in GROrbitODE.java
 *  (c) Slavomir Tuleja, 2002
 */

public class RainLight implements GROrbitODE {
   private double[] state= new double[] {10.0,0.0,0.0,0.0,0.0,0.0};
   private double a=0;
   private double invB=0.5; // invB = 1/b
   private int sign=-1;//for inward orbit sign=-1; for outward orbit sign=+1
   private double rLocalMin, rLocalMax;
   private double vmMax=1, vmMin=-1;
   //...initial state: {r,phi,pR,pPhi,tau,t}
   private GRorbits applet;

   //constructor
   public RainLight(GRorbits applet){
      this.applet=applet;
   }

   public double[] getState(){
      return state;
   }

   public void getRates(double[] state, double[] rates){
      double r,phi,pR,pPhi;
      double b,c,rho;
      double D,H,m,B1,B2,B3;
      r=state[0];
      if(r<1e-5){
         r=1e-5;
         state[0]=1e-5;
         rates[0]=0;
         rates[1]=0;
         rates[2]=0;
         rates[3]=0;
         rates[4]=0;
         rates[5]=1;
      }
      else {
         phi=state[1];
         pR=state[2];
         pPhi=state[3];

         b=Math.sqrt(r*r+a*a);
         c=Math.sqrt(2*r);
         rho=Math.sqrt(r*r);//in general rho depends on theta
         m=0;
         D=Math.sqrt( ((b*b-c*c)/(rho*rho)+b*b*c*c/(rho*rho*rho*rho))*pR*pR + 2*a*c*pR*pPhi/(b*rho*rho) + pPhi*pPhi/(b*b) + m*m );
         H=-b*c*pR/(rho*rho) + D;

         B1=(c/b)*(1 + b*b/(2*r*r) - 2*b*b/(rho*rho));
         B2=1 - c*c/(2*r*r) - (b*b-c*c)/(rho*rho) + b*c*B1/(rho*rho);
         B3=1 - b*b/(2*r*r) + 2*b*b/(rho*rho);

         rates[0]=( (b*b-c*c)*pR - b*c*H + a*c*pPhi/b )/(rho*rho*D);
         rates[1]=( a*c*pR/b + (rho*rho*pPhi)/(b*b)  )/(rho*rho*D);
         rates[2]=r*B1*pR/(rho*rho) + (r/(D*rho*rho))*(-B2*pR*pR + a*c*B3*pR*pPhi/(b*b*b) + rho*rho*pPhi*pPhi/(b*b*b*b));
         rates[3]=0;
         rates[4]=0;//1.0/D;
         rates[5]=1;
      }
   }

   public void setA(double a){
      this.a=a;
   }

   public double getA(){
      return a;
   }

   public void setIC(double r, double phi){
      //sets and computes initial conditions r, phi, pR, pPhi from
      //given vlaue of invB
      double pPhi;
      double b,c,rho;

      double H=1,m,F;//we set H=1

      //double eps=1e-10;
      //if(Math.abs(r-getRHorizon())<eps)r=getRHorizon()+eps;

      pPhi=H/invB;

      b=Math.sqrt(r*r+a*a);//b here IS NOT an impact parameter
      c=Math.sqrt(2*r);
      rho=Math.sqrt(r*r);//in general rho depends on theta
      m=0;
      F=Math.pow( (c/b)*(b*b*H-a*pPhi)/(b*b-c*c) ,2) + ((b*b-a*a)*(H*H-m*m-pPhi*pPhi/(b*b)))/(b*b-c*c);
      state[0]=r;
      state[1]=phi;
      state[2]=(c/b)*(b*b*H-a*pPhi)/(b*b-c*c) + sign*Math.sqrt(F);
      state[3]=pPhi;
      state[4]=0;
      state[5]=0;
   }

   public double getVm(double r){
      return (2*a+Math.sqrt(r*r*(a*a+r*r-2*r)))/(r*r*r+a*a*(r+2));
   }

   public double getVmAtHorizon(){
      double r;
      double b,c,rho;
      double m, H;
      H=1;
      r=getRHorizon();
      b=Math.sqrt(r*r+a*a);
      c=Math.sqrt(2*r);
      return (H/invB)*a*c*c/(r*r*b*b+a*a*c*c);
   }

   public double getRHorizon(){
      return 1+Math.sqrt(1-a*a);
   }

   public double getRInnerHorizon(){
      return 1-Math.sqrt(1-a*a);
   }

   public double getR(){
      return state[0];
   }

   public double getPhi(){
      return state[1];
   }

   public double getTau(){
      return state[4];
   }

   public Scale getScale(double rMax){
      //returns the scale object for Vm diagram
      //also it sets value of public variables rLocalMin, rLocalMax
      // if rLocalMin==0 then there's no local minimum in Vm(r)
      // else there's local minimum in Vm(r) at r=rLocalMin...
      // the same is true about rLocalMax
      double y1=0;
      double y2=1;
      double deltaY;
      double min,max,yA,yB,yC,rA,rB,rC;
      if(rMax<getRHorizon()){
         min=0;
         max=0.2;
      }
      else{
         double dr=rMax/400.0;

         min=1e14; max=-1e14;
         rLocalMin=0;
         rLocalMax=0;
         rA=getRHorizon()+1e-10; rB=rA+dr; rC=rA+2*dr;

         while (rA<=rMax-2*dr){
            double VmrA=getVm(rA);
            double VmrB=getVm(rB);
            double VmrC=getVm(rC);
            if(VmrA<min) {min=VmrA;}
            if(VmrA>max) {max=VmrA;}
            if((VmrB-VmrA)*(VmrC-VmrB)<0) {
               if((VmrB-VmrA)>0) rLocalMax=rB;
               else rLocalMin=rB;
            }
            rA+=dr;
            rB+=dr;
            rC+=dr;
         }
      }

      if(rLocalMin>getRHorizon()){
         //local minimum in Vm(r) exists...
         y1=getVm(rLocalMin);
         y2=max;
      }
      else {
         //there is no local minimum in Vm(r)...
         y1=min;
         y2=max;
      }

      setEffPotYBounds();

      if(y2>vmMax) y2=vmMax;
      if(y1<vmMin) y1=vmMin;

      deltaY=y2-y1;
      y1-=0.05*deltaY;
      y2+=0.05*deltaY;
      return new Scale(0,rMax,y1,y2,0,200,0,200);
   }

   private void setEffPotYBounds(){
      //sets vmMax and vmMin depending on applet zoom
      int zoom=applet.getZoom();
      switch(zoom){
         case 0: //x0.5
            vmMax=1;
            vmMin=0;
            return;
         case 1: //x1
            vmMax=1;
            vmMin=0;
            return;
         case 2: //x10
            vmMax=1;
            vmMin=0;
            return;
         case 3: //x100
            vmMax=1;
            vmMin=0;
            return;
         default:
            vmMax=1;
            vmMin=0;
            return;
      }
   }

   public void setSign(int s){
      sign=s;
   }

   public double getRLocalMin(double rMax){
      Scale sc=getScale(rMax);
      return rLocalMin;
   }

   public double getRLocalMax(double rMax){
      Scale sc=getScale(rMax);
      return rLocalMax;
   }

   public void setLm(double Lm){
      //does nothing...
   }

   public void setEm(double Em){
      //does nothing...
   }

   public double getOrbitParameter(){
      //returns Em or 1/b depending on ode selected...
      return invB;
   }

   public void setOrbitParameter(int j){
      invB=applet.scVm.jToy(j);;
   }

   public double getLm(){
      //this is NOT useful here...
      return 1;
   }

   public void setInvB(double invB){
      this.invB=invB;
   }

   public double getEm(){
      //this is NOT useful here...
      return 0;
   }

   public double rTorPlot(double r){
      return Math.sqrt(r*r+a*a);
   }

   public double rPlotTor(double rPlot){
      if(rPlot*rPlot>a*a) return Math.sqrt(rPlot*rPlot-a*a);
      else return 0;
   }

   public void initialSetup(){
      //scrollbars and corresponding labels...
      a=applet.slA.getValue()/1000.0;
      initialSetupTextual();
   }

   public void initialSetupTextual(){
      //scrollbars and corresponding labels...
      applet.lblA.setText("a = ".concat(MyUtils.roundOff(a,6)).concat(" M"));

      //inicialization of scale objects...
      applet.setScales(applet.getRMax());

      //adjusting invB...
      //...if invB is under Effective potential curve, we have to adjust it
      double rr=applet.getR();
      if(invB<getVm(applet.getR())) invB=getVm(applet.getR())+1e-10;

      //setting initial conditions of simulation
      setIC(applet.getR(),applet.getPhi());

      //array of orbit points
      for(int i=0; i<applet.numPoints;i++){
         applet.orbit[i]= new OrbitPoint(getR(),getPhi());
      }

      applet.repaintAll();
   }

}