/** Object encapsulating differential equations for motion of a MATERIAL
 *  PARTICLE in equatorial plane of a Schwarzschild black hole in Schwarzschild bookkeeper's Coordinates
 *  For comments about the functionality of the methods below,
 *  please see the definition of GROrbitODE interface in GROrbitODE.java
 *  (c) Slavomir Tuleja, 2002
 */

public class SchwarzschildMatter implements GROrbitODE {
   private double[] state= new double[] {10.0,0.0,0.0,0.0,0.0,0.0};
   private double a=0;
   private int sign=-1;//for inward orbit sign=-1; for outward orbit sign=+1
   private double rLocalMin, rLocalMax;
   private double Lm, Em;
   private double vmMax=5, vmMin=-5;
   //...initial state: {r,dr/dt,phi,dphi/dt,tau,t}
   private GRorbits applet;

   //constructor
   public SchwarzschildMatter(GRorbits applet){
      this.applet=applet;
   }

   public double[] getState(){
      return state;
   }

   public void getRates(double[] state, double[] rates){
      //Lagrange eqns
      double x;
      rates[0]=state[1];
      rates[1]=(((a*a-2*state[0]+state[0]*state[0])*(-1+2*a*state[3]+state[3]*state[3]*(state[0]*state[0]*state[0]-a*a)))/(state[0]*state[0]*state[0]*state[0])) - ((state[1]*state[1]*(-2*a*a+2*a*a*a*state[3]+a*a*state[0]-3*state[0]*state[0]+6*a*state[3]*state[0]*state[0]))/(state[0]*state[0]*(a*a-2*state[0]+state[0]*state[0])));
      rates[2]=state[3];
      rates[3]=-(2*state[1]*(a-2*a*a*state[3]+a*a*a*state[3]*state[3]-3*state[3]*state[0]*state[0]+3*a*state[3]*state[3]*state[0]*state[0]+state[3]*state[0]*state[0]*state[0]))/(state[0]*state[0]*(a*a-2*state[0]+state[0]*state[0]));
      x=(1-2/state[0]) + (4*a*state[3])/(state[0]) - (state[1]*state[1])/(1-2/state[0]+(a*a)/(state[0]*state[0])) -(1+(a*a)/(state[0]*state[0])+(2*a*a)/(state[0]*state[0]*state[0]))*state[0]*state[0]*state[3]*state[3];
      if(x>=0) rates[4]=Math.sqrt(x);
      else rates[4]=0;
      rates[5]=1;
   }

   public void setA(double a){
      //not useful here
      this.a=0;
   }

   public double getA(){
      //not useful here
      return a;
   }

   public void setIC(double r, double phi){
      //sets and computes initial conditions r, dr/dt, phi, dphi/dt from
      //given vlaues of energy Em and angular momentum Lm
      double eps=1e-10;
      if(Math.abs(r-getRHorizon())<eps)r=getRHorizon()+eps;

      state[0]=r;
      state[1]=sign*((state[0]*(a*a+state[0]*(state[0]-2)))*Math.sqrt((state[0]*state[0]*(2-state[0]+Em*Em*state[0])  +Lm*Lm*(2-state[0])-4*a*Em*Lm+a*a*(Em*Em*(2+state[0])-state[0]))/(state[0]*state[0]*state[0]))) / (Em*state[0]*state[0]*state[0]-2*a*Lm+a*a*Em*(2+state[0]));
      state[2]=phi;
      state[3]=(Lm*(state[0]-2)+2*a*Em)/(Em*state[0]*state[0]*state[0]-2*a*Lm+a*a*Em*(2+state[0]));
      state[4]=0;
      state[5]=0;
   }

   public double getVm(double r){
      return ((2*a*Lm)+Math.sqrt(r*(a*a-2*r+r*r)*(r*r*r+a*a*(r+2)+r*Lm*Lm)))/(r*r*r+a*a*(r+2));
   }

   public double getVmAtHorizon(){
      return a*Lm/(2.0*(1+Math.sqrt(1-a*a)));
   }

   public double getRHorizon(){
      return (1+Math.sqrt(1-a*a));
   }

   public double getRInnerHorizon(){
      return 1-Math.sqrt(1-a*a);
   }

   public double getR(){
      return state[0];
   }

   public double getPhi(){
      return state[2];
   }

   public double getTau(){
      return state[4];
   }

   public Scale getScale(double rMax){
      //returns the scale object for Vm diagram
      //also it sets value of public variables rLocalMin, rLocalMax
      // if rLocalMin==0 then there's no local minimum in Vm(r)
      // else there's local minimum in Vm(r) at r=rLocalMin...
      // the same is true about rLocalMax
      // The result depends on WHERE the orbiter is. If it is under inner Cauchy horizon, we
      // scan 0<r<rInnerHorizon. Otherwise we scan rH<r<rMax
      double y1=0;
      double y2=1;
      double deltaY;
      double min,max,yA,yB,yC,rA,rB,rC;
      double dr=rMax/400.0;
      double rStart, rStop;
      min=1e14; max=-1e14;
      rLocalMin=0;
      rLocalMax=0;

      if(rMax>getRHorizon()){
         rStart=getRHorizon();
         rStop=rMax;
      }
      else {
         rStart=0;
         rStop=getRInnerHorizon();
      }

      rA=rStart+1e-10; rB=rA+dr; rC=rA+2*dr;

      while (rA<=rStop-2*dr){
         double VmrA=getVm(rA);
         double VmrB=getVm(rB);
         double VmrC=getVm(rC);
         if(VmrA<min) {min=VmrA;}
         if(VmrA>max) {max=VmrA;}
         if((VmrB-VmrA)*(VmrC-VmrB)<0) {
            if((VmrB-VmrA)>0) rLocalMax=rB;
            else rLocalMin=rB;
         }
         rA+=dr;
         rB+=dr;
         rC+=dr;
      }


      if(rLocalMin>getRHorizon()){
         //local minimum in Vm(r) exists...
         y1=getVm(rLocalMin);
         y2=max;
      }
      else {
         //there is no local minimum in Vm(r)...
         double diff=Math.abs(Em-max);
         if(diff<0.01) diff=0.01;
         y1=max-2*diff;
         y2=max+2*diff;
      }

      setEffPotYBounds();
      if(y2>vmMax) y2=vmMax;
      if(y1<vmMin) y1=vmMin;

      if(y2<Em) y2=Em;
      if(y1>Em) y1=Em;

      deltaY=y2-y1;
      y1-=0.05*deltaY;
      y2+=0.05*deltaY;
      return new Scale(0,rMax,y1,y2,0,200,0,200);
   }

   private void setEffPotYBounds(){
      //sets vmMax and vmMin depending on applet zoom
      int zoom=applet.getZoom();
      switch(zoom){
         case 0: //x0.5
            vmMax=5;
            vmMin=-5;
            return;
         case 1: //x1
            vmMax=5;
            vmMin=-5;
            return;
         case 2: //x10
            vmMax=1;
            vmMin=0;
            return;
         case 3: //x100
            vmMax=1;
            vmMin=0;
            return;
         default:
            vmMax=5;
            vmMin=-5;
            return;
      }
   }

   public void setSign(int s){
      sign=s;
   }

   public double getRLocalMin(double rMax){
      Scale sc=getScale(rMax);
      return rLocalMin;
   }

   public double getRLocalMax(double rMax){
      Scale sc=getScale(rMax);
      return rLocalMax;
   }

   public void setLm(double Lm){
      this.Lm=Lm;
   }

   public void setInvB(double invB){
      //does nothing here ...
   }

   public void setEm(double Em){
      this.Em=Em;
   }

   public double getOrbitParameter(){
      //returns Em or 1/b depending on ode selected...
      return Em;
   }

   public void setOrbitParameter(int j){
      Em=applet.scVm.jToy(j);
   }

   public double getLm(){
      return Lm;
   }

   public double getEm(){
      return Em;
   }

   public double rTorPlot(double r){
      return Math.sqrt(r*r+a*a);
   }

   public double rPlotTor(double rPlot){
      if(rPlot*rPlot>a*a) return Math.sqrt(rPlot*rPlot-a*a);
      else return 0;
   }

   public void initialSetup(){
      //scrollbars and corresponding labels...
     Lm=applet.slLm.getValue()/1000.0;
     a=applet.slA.getValue()/1000.0;
     initialSetupTextual();
   }

   public void initialSetupTextual(){
      //scrollbars and corresponding labels...
      applet.lblLm.setText("L/m = ".concat(MyUtils.roundOff(Lm,6)).concat(" M"));
      applet.lblA.setText("a = ".concat(MyUtils.roundOff(a,6)).concat(" M"));

      //inicialization of scale objects...
      applet.setScales(applet.getRMax());

      //adjusting energy...
      //...if Em is under Effective potential curve, we have to adjust it
      if(Em<getVm(applet.getR())) Em=getVm(applet.getR())+1e-10;

      //setting initial conditions of simulation
      setIC(applet.getR(),applet.getPhi());

      //array of orbit points
      for(int i=0; i<applet.numPoints;i++){
         applet.orbit[i]= new OrbitPoint(getR(),getPhi());
      }

      applet.repaintAll();
   }

}